<?php

namespace Mbe\Shipping\Model;

use Magento\Shipping\Model\Carrier\AbstractCarrier;
use Magento\Shipping\Model\Carrier\CarrierInterface;


/**
 * Created by PhpStorm.
 * User: Samuele Carpene
 * Web: http://www.samuelecarpene.com
 * Date: 31/08/2016
 * Time: 14:33
 */
class Carrier
    extends AbstractCarrier
    implements CarrierInterface
{

    const SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_ITEM = 1;
    const SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_SINGLE_PARCEL = 2;
    const SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_MULTI_PARCEL = 3;

    const HANDLING_TYPE_PER_SHIPMENT = "S";
    const HANDLING_TYPE_PER_PARCEL = "P";


    /**
     * Carrier's code, as defined in parent class
     *
     * @var string
     */
    protected $_code = 'mbe_shipping';

    /** @var $logger Mbe_Shipping_Helper_Logger */
    protected $logger = null;
    /** @var $helper Mbe_Shipping_Helper_Data */
    protected $helper = null;

    /**
     * @var \Mbe\Shipping\Helper\Data
     */
    protected $shippingHelper;

    /**
     * @var \Mbe\Shipping\Helper\Logger
     */
    protected $shippingLoggerHelper;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Shipping\Model\Rate\ResultFactory
     */
    protected $shippingRateResultFactory;

    /**
     * @var \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory
     */
    protected $quoteQuoteAddressRateResultErrorFactory;

    /**
     * @var \Mbe\Shipping\Model\WsFactory
     */
    protected $shippingWsFactory;

    /**
     * @var \Mbe\Shipping\Helper\Rates
     */
    protected $shippingRatesHelper;
    
    /**
     * @var \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory
     */
    protected $quoteQuoteAddressRateResultMethodFactory;

    /**
     * @var \Mbe\Shipping\Helper\Tracking
     */
    protected $shippingTrackingHelper;

    /**
     * @var \Magento\Shipping\Model\Tracking\Result\StatusFactory
     */
    protected $shippingTrackingResultStatusFactory;

    public function __construct(
        \Mbe\Shipping\Helper\Data $shippingHelper,
        \Mbe\Shipping\Helper\Logger $shippingLoggerHelper,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Shipping\Model\Rate\ResultFactory $shippingRateResultFactory,
        \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory $quoteQuoteAddressRateResultErrorFactory,
        \Mbe\Shipping\Model\WsFactory $shippingWsFactory,
        \Mbe\Shipping\Helper\Rates $shippingRatesHelper,
        \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory $quoteQuoteAddressRateResultMethodFactory,
        \Mbe\Shipping\Helper\Tracking $shippingTrackingHelper,
        \Magento\Shipping\Model\Tracking\Result\StatusFactory $shippingTrackingResultStatusFactory
    ) {
        $this->shippingHelper = $shippingHelper;
        $this->shippingLoggerHelper = $shippingLoggerHelper;
        $this->scopeConfig = $scopeConfig;
        $this->shippingRateResultFactory = $shippingRateResultFactory;
        $this->quoteQuoteAddressRateResultErrorFactory = $quoteQuoteAddressRateResultErrorFactory;
        $this->shippingWsFactory = $shippingWsFactory;
        $this->shippingRatesHelper = $shippingRatesHelper;
        $this->quoteQuoteAddressRateResultMethodFactory = $quoteQuoteAddressRateResultMethodFactory;
        $this->shippingTrackingHelper = $shippingTrackingHelper;
        $this->shippingTrackingResultStatusFactory = $shippingTrackingResultStatusFactory;
    }

    public function getConfigData($field)
    {
        if (empty($this->_code)){
            return false;
        }

        $path = 'carriers/' . $this->_code . '/' . $field;
        
        return $this->scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->getStore());
    }

    public function getRequestWeight(\Magento\Quote\Model\Quote\Address\RateRequest $request)
    {
        /*
        $result = 0;
        foreach ($request->getAllItems() as $item) {
            $result += $item->getWeight();
        }
        return $result;
        */
        return $request->getPackageWeight();
    }

    /**
     * Returns available shipping rates for Inchoo Shipping carrier
     *
     * @param \Magento\Quote\Model\Quote\Address\RateRequest $request
     * @return \Magento\Shipping\Model\Rate\Result
     */
    public function collectRates(\Magento\Quote\Model\Quote\Address\RateRequest $request)
    {

        $this->shippingLoggerHelper->log('collectRates');


        //TODO: check countries
        /**
         * @var $helper Mbe_Shipping_Helper_Data
         * @var $request Mage_Shipping_Model_Rate_Request
         */
        $helper = $this->shippingHelper;


        if (!$helper->isEnabled($this->getStoreId())){
            $this->shippingLoggerHelper->log('module disabled');
            return false;
        }
        
        $this->shippingLoggerHelper->log('module enabled');


        /** @var \Magento\Shipping\Model\Rate\Result $result */
        $result = $this->shippingRateResultFactory->create();

        $destCountry = $request->getDestCountryId();
        $destRegion = $request->getDestRegionCode();
        //TODO:fix this
        $destCity = "";
        
        $destRegionId = $request->getDestRegionId();

        $destPostCode = $request->getDestPostcode();

        $this->shippingLoggerHelper->log("Destination: COUNTRY: " . $destCountry . " - REGION ID: " . $destRegionId . " - REGION CODE: " . $destRegion . " - POSTCODE: " . $destPostCode);


        $shipmentConfigurationMode = $helper->getShipmentConfigurationMode();


        $shipments = array();
        $baseSubtotalInclTax = $request->getBaseSubtotalInclTax();

        $this->shippingLoggerHelper->log("SHIPMENTCONFIGURATIONMODE: " . $shipmentConfigurationMode);

        if ($shipmentConfigurationMode == self::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_ITEM) {
            /** @var $item Mage_Sales_Model_Quote_Item */
            $iteration = 1;
            foreach ($request->getAllItems() as $item) {
                if ($item->getParentItemId() == null) {
                    for ($i = 1; $i <= $item->getQty(); $i++) {
                        $this->shippingLoggerHelper->log("Product Iteration: " . $iteration);
                        $weight = $item->getWeight();

                        if ($helper->getShipmentsInsuranceMode() == \Mbe\Shipping\Helper\Data::MBE_INSURANCE_WITH_TAXES) {
                            $insuranceValue = $item->getRowTotalInclTax() / $item->getQty();
                        }
                        else {
                            $insuranceValue = $item->getRowTotal() / $item->getQty();
                        }

                        $shipments = $this->getRates(
                            $destCountry, $destRegion, $destCity, $destPostCode, $baseSubtotalInclTax, $weight, 1, $shipments, $iteration, $insuranceValue
                        );

                        $iteration++;
                    }
                }
            }
        }
        elseif ($shipmentConfigurationMode == self::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_SINGLE_PARCEL) {
            $maxPackageWeight = $helper->getMaxPackageWeight();
            $boxesWeights = array();
            $insuranceValue = 0;

            foreach ($request->getAllItems() as $item) {
                if ($item->getParentItemId() == null) {
                    if ($helper->getShipmentsInsuranceMode() == \Mbe\Shipping\Helper\Data::MBE_INSURANCE_WITH_TAXES) {
                        $insuranceValue += $item->getRowTotalInclTax();
                    }
                    else {
                        $insuranceValue += $item->getRowTotal();
                    }

                    for ($i = 1; $i <= $item->getQty(); $i++)
                    {
                        $canAddToExistingBox = false;
                        for ($j = 0; $j < count($boxesWeights); $j++) {
                            $newWeight = $boxesWeights[$j] + $item->getWeight();
                            if ($newWeight < $maxPackageWeight) {
                                $canAddToExistingBox = true;
                                $boxesWeights[$j] = $newWeight;
                                break;
                            }
                        }
                        if (!$canAddToExistingBox) {
                            $boxesWeights[] = $item->getWeight();
                        }
                    }
                }
            }

            $numBoxes = count($boxesWeights);

            $this->shippingLoggerHelper->log("Num Boxes: " . $numBoxes);

            $shipments = $this->getRates(
                $destCountry, $destRegion, $destCity, $destPostCode, $baseSubtotalInclTax, $boxesWeights, $numBoxes, array(), 1, $insuranceValue
            );
        }
        elseif ($shipmentConfigurationMode == self::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_MULTI_PARCEL) {
            $boxesWeights = array();
            $numBoxes = 0;
            $insuranceValue = 0;
            
            foreach ($request->getAllItems() as $item) {
                if ($item->getParentItemId() == null) {
                    if ($helper->getShipmentsInsuranceMode() == \Mbe\Shipping\Helper\Data::MBE_INSURANCE_WITH_TAXES) {
                        $insuranceValue += $item->getRowTotalInclTax();
                    }
                    else {
                        $insuranceValue += $item->getRowTotal();
                    }

                    $numBoxes += $item->getQty();
                    for ($i = 1; $i <= $item->getQty(); $i++) {
                        $boxesWeights[] = $item->getWeight();
                    }
                }
            }

            $this->shippingLoggerHelper->log("Num Boxes: " . $numBoxes);
            $shipments = $this->getRates(
                $destCountry, $destRegion, $destCity, $destPostCode, $baseSubtotalInclTax, $boxesWeights, $numBoxes, array(), 1, $insuranceValue
            );
            
        }


        //TODO- remove subzone if it is the same for all shipping methods

        $subZones = array();
        foreach ($shipments as $shipment) {
            if (!in_array($shipment->subzone_id, $subZones)) {
                array_push($subZones, $shipment->subzone_id);
            }
        }

        $useSubZone = false;
        if (count($subZones) > 1) {
            $useSubZone = true;
        }

        if (empty($shipments)) {
            $errorTitle = 'Unable to retrieve shipping methods';
            $error = $this->quoteQuoteAddressRateResultErrorFactory->create();
            $error->setCarrier($this->_code);
            $error->setCarrierTitle($this->getConfigData('title'));
            $error->setErrorMessage($errorTitle);
            $error->setErrorMessage($this->getConfigData('specificerrmsg'));
            $result->append($error);
        }
        else {

            foreach ($shipments as $shipment) {
                if ($useSubZone) {
                    $currentRate = $this->_getRate($shipment->title_full, $shipment->shipment_code, $shipment->price);
                }
                else {
                    $currentRate = $this->_getRate($shipment->title, $shipment->shipment_code, $shipment->price);
                }

                $result->append($currentRate);
            }
        }

        return $result;
    }


    private function getRates(
        $destCountry, $destRegion, $city, $destPostCode, $baseSubtotalInclTax, $weight, $boxes, $oldResults = array(), $iteration = 1, $insuranceValue
    )
    {
        $this->shippingLoggerHelper->log("getRates");

        /** @var $helper Mbe_Shipping_Helper_Data */
        $helper = $this->shippingHelper;

        /** @var $ws Mbe_Shipping_Model_Ws */
        $ws = $this->shippingWsFactory->create();

        $result = array();
        $newResults = array();

        $ratesHelper = $this->shippingRatesHelper;
        
        if ($ratesHelper->useCustomRates($destCountry)) {
            $totalWeight = $weight;
            if (is_array($weight)) {
                $totalWeight = 0;
                foreach ($weight as $singleWeight) {
                    $totalWeight += $singleWeight;
                }
            }
            $shipments = $ratesHelper->getCustomRates($destCountry, $destRegion, $city, $destPostCode, $totalWeight, $insuranceValue);
        }
        else {
            $shipments = $ws->estimateShipping($destCountry, $destRegion, $destPostCode, $weight, $boxes, $insuranceValue);
        }

        $this->shippingLoggerHelper->logVar($shipments, 'ws estimateShipping result');

        if ($shipments) {

            $allowedShipmentServicesArray = $helper->getAllowedShipmentServicesArray();

            foreach ($shipments as $shipment) {
                //TODO: check if is sufficient use service
                $shipmentMethod = $shipment->Service;

                $shipmentMethodKey = $shipment->Service . "_" . $shipment->IdSubzone;

                if (in_array($shipmentMethod, $allowedShipmentServicesArray)) {
                    $shipmentTitle = __($shipment->ServiceDesc);
                    $shipmentTitle .= " - " . __($shipment->SubzoneDesc);

                    $shipmentPrice = $shipment->NetShipmentTotalPrice;

                    $shipmentPrice = $this->applyFee($shipmentPrice, $boxes);


                    $shippingThreshold = $helper->getThresholdByShippingServrice($shipmentMethod);
                    //var_dump($shippingThreshold);
                    if ($shippingThreshold != null && $baseSubtotalInclTax >= $shippingThreshold)
                    {
                        $shipmentPrice = 0;
                    }

                    //TODO: check if is necessary to save in some mode the courier data

                    $current = new \stdClass();
                    $current->title = __($shipment->ServiceDesc);
                    $current->title_full = $shipmentTitle;
                    $current->method = $shipmentMethod;
                    $current->price = $shipmentPrice;
                    $current->subzone = $shipment->SubzoneDesc;
                    $current->subzone_id = $shipment->IdSubzone;
                    $current->shipment_code = $shipmentMethodKey;


                    $newResults[$shipmentMethodKey] = $current;
                }
            }

            if ($iteration == 1){
                $result = $newResults;
            }
            else {
                foreach ($newResults as $newResultKey => $newResult) {
                    if (array_key_exists($newResultKey, $oldResults)) {
                        $newResult->price += $oldResults[$newResultKey]->price;
                        $result[$newResultKey] = $newResult;
                    }

                }
            }

        }

        return $result;
    }

    /**
     * Returns Allowed shipping methods
     *
     * @return array
     */
    public function getAllowedMethods()
    {
        return array(
            'standard'  => 'Standard delivery',
            'express'   => 'Express delivery',
        );
    }

    //
    /**
     * Get rate object based on name and price
     *
     * @param $title string Title name
     * @param $method string Method name
     * @param $price float Shipping Method price
     * @return \Magento\Quote\Model\Quote\Address\RateResult\Method
     */
    protected function _getRate($title, $method, $price)
    {
        /** @var $helper Mbe_Shipping_Helper_Data */
        $helper = $this->shippingHelper;
        $price = $helper->round($price);

        /** @var $rate Mage_Shipping_Model_Rate_Result_Method */
        $rate = $this->quoteQuoteAddressRateResultMethodFactory->create();
        $rate->setCarrier($this->_code);
        $rate->setCarrierTitle(__($this->getConfigData('title')));
        $rate->setMethod($method);
        $rate->setMethodTitle($title);
        $rate->setPrice($price);
        $rate->setCost(0);
        return $rate;
    }


    public function applyFee($value, $packages = 1)
    {
        $handlingType = $this->getConfigData("handling_type");
        $handlingAction = $this->getConfigData("handling_action");
        $handlingFee = $this->getConfigData("handling_fee");

        if ($handlingAction == self::HANDLING_TYPE_PER_SHIPMENT) {
            $packages = 1;
        }

        if (self::HANDLING_TYPE_FIXED == $handlingType) {
            //fixed
            $result = $value + $handlingFee * $packages;
        }
        else {
            //percent
            $result = $value * (100 + $handlingFee) / 100;
        }

        return $result;
    }

    public function isTrackingAvailable()
    {
        return true;
    }

    public function getTrackingInfo($tracking)
    {

        $helper = $this->shippingHelper;
        $trackingHelper = $this->shippingTrackingHelper;
        $mbeUrl = $trackingHelper->getTrackingUrlBySystem($helper->getCountry());

        $track = $this->shippingTrackingResultStatusFactory->create();


        $track->setUrl($mbeUrl . $tracking)
            ->setTracking($tracking)
            ->setCarrierTitle($this->getConfigData('title'));
        return $track;
    }
}