<?php

namespace Mbe\Shipping\Model;

use Magento\Framework\Event\ObserverInterface;
use Magento\Sales\Model\Order\Shipment\NotifierInterface;
use Magento\Sales\Model\Order\Shipment\TrackRepository;
use Magento\Sales\Model\Order\ShipmentRepository;

class Observer implements ObserverInterface
{

    /**
     * @var \Mbe\Shipping\Helper\Data
     */
    protected $shippingHelper;

    /**
     * @var \Mbe\Shipping\Helper\Logger
     */
    protected $shippingLoggerHelper;

    /**
     * @var \Magento\Sales\Model\OrderFactory
     */
    protected $salesOrderFactory;

    /**
     * @var \Magento\Framework\DB\TransactionFactory
     */
    protected $transactionFactory;

    /**
     * @var \Mbe\Shipping\Model\WsFactory
     */
    protected $shippingWsFactory;

    /**
     * @var \Magento\Sales\Model\Order\Shipment\TrackFactory
     */
    protected $salesOrderShipmentTrackFactory;

    /**
     * @var \Magento\Framework\Stdlib\DateTime\DateTimeFactory
     */
    protected $dateTimeDateTimeFactory;

    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\CollectionFactory
     */
    protected $salesResourceModelOrderCollectionFactory;

    protected $_objectManager;
    protected $request;
    protected $notifier;
    protected $shipmentRepository;
    protected $trackRepository;

    public function __construct(
        \Mbe\Shipping\Helper\Data $shippingHelper,
        \Mbe\Shipping\Helper\Logger $shippingLoggerHelper,
        \Magento\Sales\Model\OrderFactory $salesOrderFactory,
        \Magento\Framework\DB\TransactionFactory $transactionFactory,
        \Mbe\Shipping\Model\WsFactory $shippingWsFactory,
        \Magento\Sales\Model\Order\Shipment\TrackFactory $salesOrderShipmentTrackFactory,
        \Magento\Framework\Stdlib\DateTime\DateTimeFactory $dateTimeDateTimeFactory,
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $salesResourceModelOrderCollectionFactory,
        \Magento\Framework\ObjectManagerInterface $objectManager,
        \Magento\Framework\HTTP\PhpEnvironment\Request $request,
        NotifierInterface $notifier,
        ShipmentRepository $shipmentRepository,
        TrackRepository $trackRepository

    ) {
        $this->shippingHelper = $shippingHelper;
        $this->shippingLoggerHelper = $shippingLoggerHelper;
        $this->salesOrderFactory = $salesOrderFactory;
        $this->transactionFactory = $transactionFactory;
        $this->shippingWsFactory = $shippingWsFactory;
        $this->salesOrderShipmentTrackFactory = $salesOrderShipmentTrackFactory;
        $this->dateTimeDateTimeFactory = $dateTimeDateTimeFactory;
        $this->salesResourceModelOrderCollectionFactory = $salesResourceModelOrderCollectionFactory;
        $this->_objectManager = $objectManager;
        $this->request = $request;
        $this->notifier = $notifier;
        $this->shipmentRepository = $shipmentRepository;
        $this->trackRepository = $trackRepository;
    }

    /**
     * Check if order need to create mbe shipping
     *
     * @param \Magento\Framework\Event\Observer $observer
     * @return $this
     */
    public function execute(\Magento\Framework\Event\Observer $observer)
    {
        /**@var $helper Mbe_Shipping_Helper_Data */
        $helper = $this->shippingHelper;
        if ($helper->isEnabled()) {
            $order = $observer->getOrder();
            //only if order is shipped with mbe
            if ($helper->isMbeShipping($order)) {
                $stateNew = \Magento\Sales\Model\Order::STATE_NEW;
                $stateProcessing = \Magento\Sales\Model\Order::STATE_PROCESSING;
                $paymentMethod = $order->getPayment()->getMethodInstance()->getCode();

//                Order has been set to processing
                if ($order->getState() == $stateProcessing && $order->getOrigData('state') != $stateProcessing && $order->canShip()) {
                    $this->createShipment($order);
                }

                if ($order->getState() == $stateNew && $paymentMethod == "cashondelivery") {
                    $this->createShipment($order);
                }

                // Shipment manual creation or invoice+ship or mass action
                if ($observer->getDataByKey('mbeShipment')) {
                    $this->createShipment($order, $observer->getDataByKey('comment'));
                }
            }
        }
        return $this;
    }

    private function createShipment(\Magento\Sales\Model\Order $o, $comment = null)
    {
        /** @var  $logger \Mbe\Shipping\Helper\Logger */
        $logger = $this->shippingLoggerHelper;
        /** @var $helper \Mbe\Shipping\Helper\Data */
        $helper = $this->shippingHelper;

        $logger->log("CREATESHIPMENT");

//        $order = $this->salesOrderFactory->create()->load($o->getId());
        $order = $o;

        $shippingMethod = $helper->getShippingMethod($order);

        $insurance = $helper->isShippingWithInsurance($shippingMethod);
        if ($insurance) {
            $shippingMethod = $helper->convertShippingCodeWithoutInsurance($shippingMethod);
        }

        $isCod = false;
        $paymentMethod = $order->getPayment()->getMethodInstance()->getCode();

        if ($paymentMethod == "cashondelivery") {
            $isCod = true;
        }

        $service = $helper->getShippingCodeFromMagentoShippingMethod($shippingMethod);
        $subzone = $helper->getShippingSubZoneFromMagentoShippingMethod($shippingMethod);

        $orderTotal = $order->getGrandTotal();

        if ($order->canShip()) {
            $shipmentConfigurationMode = $helper->getShipmentConfigurationMode();

            if ($shipmentConfigurationMode == \Mbe\Shipping\Model\Carrier::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_ITEM) {
                $productsAmount = 0;
                foreach ($order->getAllItems() as $item) {
                    $productsAmount += $item->getQtyOrdered();
                }

                $codValue = $orderTotal / $productsAmount;

                foreach ($order->getAllItems() as $item) {
                    if ($item->getParentItemId() == null) {
                        $itemQty = $item->getQtyOrdered()
                            - $item->getQtyShipped()
                            - $item->getQtyRefunded()
                            - $item->getQtyCanceled();

                        $logger->log("Quantità oggetti " . $itemQty);

                        $products = [];
                        $p = new \stdClass();
                        $p->SKUCode = $item->getSku();
                        $p->Description = $helper->getProductTitleFromOrderItem($item);
                        $p->Quantity = 1;
                        array_push($products, $p);

                        $subTotal = $item->getRowTotal();
                        $subTotalInclTax = $item->getRowTotalInclTax();
                        $goodsValue = ($subTotal) / $item->getQtyOrdered();

                        if ($helper->getShipmentsInsuranceMode() == \Mbe\Shipping\Helper\Data::MBE_INSURANCE_WITH_TAXES) {
                            $insuranceValue = ($subTotalInclTax) / $item->getQtyOrdered();
                        } else {
                            $insuranceValue = ($subTotal) / $item->getQtyOrdered();
                        }

                        $itemWeight = $item->getWeight();
                        for ($i = 1; $i <= $itemQty; $i++) {
                            $qty = [];
                            $qty[$item->getId()] = 1;

                            $logger->logVar($qty, 'qty');
                            $this->createSingleShipment(
                                $order,
                                $service,
                                $subzone,
                                $itemWeight,
                                $products,
                                1,
                                $qty,
                                $goodsValue,
                                $isCod,
                                $codValue,
                                $insurance,
                                $insuranceValue
                            );
                        }
                    }
                }
            } elseif ($shipmentConfigurationMode == \Mbe\Shipping\Model\Carrier::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_SINGLE_PARCEL) {
                $maxPackageWeight = $helper->getMaxPackageWeight();
                $boxesWeights = [];
                $products = [];
                $goodsValue = 0.0;
                $insuranceValue = 0.0;
                $codValue = $orderTotal;

                foreach ($order->getAllItems() as $item) {
                    if ($item->getParentItemId() == null) {
                        $itemQty = $item->getQtyOrdered();

                        $p = new \stdClass();
                        $p->SKUCode = $item->getSku();
                        $p->Description = $helper->getProductTitleFromOrderItem($item);
                        $p->Quantity = $itemQty;
                        array_push($products, $p);

                        for ($i = 1; $i <= $item->getQtyOrdered(); $i++) {
                            $canAddToExistingBox = false;

                            for ($j = 0; $j < count($boxesWeights); $j++) {
                                $newWeight = $boxesWeights[$j] + $item->getWeight();
                                if ($newWeight < $maxPackageWeight) {
                                    $canAddToExistingBox = true;
                                    $boxesWeights[$j] = $newWeight;
                                    break;
                                }
                            }

                            if (!$canAddToExistingBox) {
                                $boxesWeights[] = $item->getWeight();
                            }
                            $goodsValue += $item->getPrice();
                        }

                        $subTotal = $item->getRowTotal();
                        $subTotalInclTax = $item->getRowTotalInclTax();
                        if ($helper->getShipmentsInsuranceMode() == \Mbe\Shipping\Helper\Data::MBE_INSURANCE_WITH_TAXES) {
                            $insuranceValue += $subTotalInclTax;
                        } else {
                            $insuranceValue += $subTotal;
                        }
                    }
                }

                $numBoxes = count($boxesWeights);
                $logger->logVar($numBoxes, "boxes amount");
                $logger->logVar($boxesWeights, "boxes weights");
                $this->createSingleShipment(
                    $order,
                    $service,
                    $subzone,
                    $boxesWeights,
                    $products,
                    $numBoxes,
                    [],
                    $goodsValue,
                    $isCod,
                    $codValue,
                    $insurance,
                    $insuranceValue
                );
            } elseif ($shipmentConfigurationMode == \Mbe\Shipping\Model\Carrier::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_MULTI_PARCEL) {
                $boxesWeights = [];
                $numBoxes = 0;
                $products = [];
                $goodsValue = 0.0;
                $insuranceValue = 0.0;
                $codValue = $orderTotal;

                foreach ($order->getAllItems() as $item) {
                    if ($item->getParentItemId() == null) {
                        $itemQty = $item->getQtyOrdered();
                        $numBoxes += $item->getQtyOrdered();

                        $p = new \stdClass();
                        $p->SKUCode = $item->getSku();
                        $p->Description = $helper->getProductTitleFromOrderItem($item);
                        $p->Quantity = $itemQty;
                        array_push($products, $p);

                        for ($i = 1; $i <= $itemQty; $i++) {
                            $boxesWeights[] = $item->getWeight();
                            $goodsValue += $item->getPrice();
                        }

                        $subTotal = $item->getRowTotal();
                        $subTotalInclTax = $item->getRowTotalInclTax();

                        if ($helper->getShipmentsInsuranceMode() == \Mbe\Shipping\Helper\Data::MBE_INSURANCE_WITH_TAXES) {
                            $insuranceValue += $subTotalInclTax;
                        } else {
                            $insuranceValue += $subTotal;
                        }
                    }
                }

                $logger->logVar($numBoxes, "boxes amount");
                $logger->logVar($boxesWeights, "boxes weights");
                $this->createSingleShipment(
                    $order,
                    $service,
                    $subzone,
                    $boxesWeights,
                    $products,
                    $numBoxes,
                    [],
                    $goodsValue,
                    $isCod,
                    $codValue,
                    $insurance,
                    $insuranceValue
                );
            }
        }
    }

    public function createSingleShipment($order, $service, $subzone, $weight, $products, $boxes, $qty = [], $goodsValue = 0.0, $isCod = false, $codValue = 0.0, $insurance, $insuranceValue, $comment = null)
    {
        $sendEmail = true;
        $includeComment = true;
        $comment = ($comment ? ($comment . ", ") : "") . "Shipment created through MBE";

        $helper = $this->shippingHelper;
        $logger = $this->shippingLoggerHelper;

        /* prepare to create shipment */
        $shipment = $this->prepareShipment($order, $qty);

        if ($shipment) {
            $shipment->register();
            $shipment->addComment($comment, $sendEmail && $includeComment);
            // add manually insert custom comment
            $postComment = $this->request->getParam('shipment', []);
            $comment = isset($postComment['comment_text']) ? filter_var($postComment['comment_text'], FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH) : '';
            if (!empty($comment)) {
                $shipment->addComment($comment);
            }
            $shipment->getOrder()->setIsInProcess(true);

            try {
                $transactionSave = $this->transactionFactory->create()
                    ->addObject($shipment)
                    ->addObject($shipment->getOrder())
                    ->save();
                //TODO: create MBE SHIPMENT

                $fileName = $shipment->getIncrementId();

                $ws = $this->shippingWsFactory->create();

                /*
                $shippingMethod = $order->getShippingMethod();

                $insurance = $helper->isShippingWithInsurance($shippingMethod);
                if($insurance){
                    $shippingMethod = $helper->convertShippingCodeWithoutInsurance($shippingMethod);
                }

                $service = $helper->getShippingCodeFromMagentoShippingMethod($shippingMethod);
                $subzone = $helper->getShippingSubZoneFromMagentoShippingMethod($shippingMethod);
                */

                $shippingAddress = $order->getShippingAddress();
                $firstName = $shippingAddress->getFirstname();
                $lastName = $shippingAddress->getLastname();
                $companyName = $shippingAddress->getCompany();
                $addressArray = $shippingAddress->getStreet();
                $address = implode(" ", $addressArray);
                $phone = $shippingAddress->getTelephone();
                $city = $shippingAddress->getCity();
                $countryId = $shippingAddress->getCountryId();
                /*
                $region = $shippingAddress->getRegionId();
                if (!$region) {
                    $region = $shippingAddress->getRegion();
                }
                */
                $region = $shippingAddress->getRegion();
                $postCode = $shippingAddress->getPostcode();
                $email = $shippingAddress->getEmail();

                $orderIncrementId = $order->getIncrementId();
                $reference = $orderIncrementId;

                $uapShipment = ($this->shippingHelper->getShipToUap() && $order->getmbeShippingUapShipment());

                $mbeShipment = $ws->createShipping(
                    $countryId,
                    $region,
                    $postCode,
                    $weight,
                    $boxes,
                    $products,
                    $service,
                    $subzone,
                    $firstName,
                    $lastName,
                    $companyName,
                    $address,
                    $phone,
                    $city,
                    $email,
                    $uapShipment,
                    $goodsValue,
                    $reference,
                    $isCod,
                    $codValue,
                    $insurance,
                    $insuranceValue,
                    $comment
                );

                $logger->logVar($mbeShipment, "MBE SHIPMENT");

                $trackingNumber = $mbeShipment->MasterTrackingMBE;
                $label = $mbeShipment->Labels->Label;
                if (is_array($label)) {
                    $i = 1;
                    foreach ($label as $l) {
                        $this->saveShipmentDocument($l->Type, $l->Stream, $fileName . '_' . $i);
                        $i++;
                    }
                } else {
                    $this->saveShipmentDocument($label->Type, $label->Stream, $fileName);
                }

                $track = $this->salesOrderShipmentTrackFactory->create()
                    ->setShipment($shipment)
                    ->setData('title', 'Mbe Shipping')
                    ->setData('track_number', $trackingNumber)
                    ->setData('carrier_code', 'mbe_shipping')
                    ->setData('order_id', $shipment->getData('order_id'))
                    ->setData('parent_id', $shipment->getData('entity_id'));
                $this->trackRepository->save($track);

                $this->notifier->notify($order, $shipment);
                $this->shipmentRepository->save($shipment);
            } catch (\Exception $e) {
                $logger->log($e->getMessage());
            }
        }
    }

    public function prepareShipment($order, $qtys = [])
    {
        $totalQty = 0;
        $convertOrder = $this->_objectManager->create('Magento\Sales\Model\Convert\Order');
        $shipment = $convertOrder->toShipment($order);

        foreach ($order->getAllItems() as $orderItem) {
            if (!$this->canShipItem($orderItem, $qtys)) {
                continue;
            }

            // convert order item to a shipment item
            $shipmentItem = $convertOrder->itemToShipmentItem($orderItem);

            if (isset($qtys[$orderItem->getId()])) {
                $qty = min($qtys[$orderItem->getId()], $orderItem->getQtyToShip());
            } elseif (!count($qtys)) {
                $qty = $orderItem->getQtyToShip();
            } else {
                continue;
            }

            $totalQty += $qty;
            $shipmentItem->setQty($qty);
            $shipment->addItem($shipmentItem);
        }
        $shipment->setTotalQty($totalQty);
        return $shipment;
    }

    protected function canShipItem($item, $qtys = [])
    {
        if ($item->getIsVirtual() || $item->getLockedDoShip()) {
            return false;
        }

        return $item->getQtyToShip() > 0;
    }

    /** Method that save shipment document
     * @param $type String document type
     * @param $content String document content
     * @param $filename String filename for new document to save
     */
    public function saveShipmentDocument($type, $content, $filename)
    {
        /**@var $helper Mbe_Shipping_Helper_Data */
        $helper = $this->shippingHelper;

        /**@var $logger Mbe_Shipping_Helper_Logger */
        $logger = $this->shippingLoggerHelper;

        $ext = "txt";
        if ($type == "HTML") {
            $ext = "html";
        } elseif ($type == "PDF") {
            $ext = "pdf";
        } elseif ($type == "GIF") {
            $ext = "gif";
        }

        $filePath = $helper->getShipmentFilePath($filename, $ext);
        $saveResult = file_put_contents($filePath, $content);

        $message = "Saving shipping document :" . $filePath;

        if ($saveResult) {
            $message .= " OK";
        } else {
            $message .= " FAILURE";
        }

        $logger->log($message);
    }

    // prepareShipment manually, a replacement for Magento 1.9 $order->prepareShipment

    /**
     * Method called from magento cron based on module configuration
     */
    public function executeCron()
    {
        /**@var $helper Mbe_Shipping_Helper_Logger */
        $logger = $this->shippingLoggerHelper;
        /**@var $helper Mbe_Shipping_Helper_Data */
        $helper = $this->shippingHelper;

        $time = $this->dateTimeDateTimeFactory->create()->date('H:i');
        $logger->log("Cron Execution at " . $time);
        if ($helper->isEnabled()) {
            $this->closeShipments();
        } else {
            $logger->log("Cron Execution stopped because module is disabled or user not active");
        }
    }

    /**
     * Method that close all not closed shipments
     */
    public function closeShipments()
    {
        /**@var $helper Mbe_Shipping_Helper_Data */
        $helper = $this->shippingHelper;
        /**@var $logger Mbe_Shipping_Helper_Logger */
        $logger = $this->shippingLoggerHelper;
        /** @var  $ws Mbe_Shipping_Model_Ws */
        $ws = $this->shippingWsFactory->create();

        if ($ws->mustCloseShipments()) {
            if ($helper->getShipmentsClosureMode() == \Mbe\Shipping\Helper\Data::MBE_CLOSURE_MODE_AUTOMATICALLY) {
                $time = time();
                $to = date('Y-m-d H:i:s', $time);
                //$lastTime = $time - 172800; // 60*60*24*2
                //TODO: check if decrease this value
                $lastTime = $time - 60 * 60 * 24 * 30; // 60*60*24*2
                $from = date('Y-m-d H:i:s', $lastTime);

                $orderCollection = $this->salesResourceModelOrderCollectionFactory->create();
                $stateComplete = \Magento\Sales\Model\Order::STATE_COMPLETE;
                $orderCollection
                    ->addFieldToFilter('state', $stateComplete)
                    ->addAttributeToSelect('entity_id')
                    //->addAttributeToSelect('increment_id')
                    ->addAttributeToSelect('created_at')
                    ->addAttributeToFilter('created_at', ['from' => $from, 'to' => $to])
                    ->load();

                foreach ($orderCollection as $o) {
                    $order = $this->salesOrderFactory->create()->load($o->getId());
                    $orderIncrementId = $order->getIncrementId();

                    if ($helper->isMbeShipping($order)) {
                        //TODO: get shipments and close theme if theme are opened

                        foreach ($order->getShipmentsCollection() as $shipment) {
                            $shipmentIncrementId = $shipment->getIncrementId();

                            $tracks = $shipment->getAllTracks();
                            foreach ($tracks as $track) {
                                $trackingNumber = $track->getTrackNumber();

                                if ($helper->isTrackingOpen($trackingNumber)) {
                                    $msg = "Order: " . $orderIncrementId . " - Shipment: " . $shipmentIncrementId . " is open.";
                                    $logger->log($msg);

                                    $ws->closeTrackingNumber($trackingNumber);
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}
