<?php

namespace Mbe\Shipping\Plugin\Model\Config\Structure;

use Magento\Framework\Stdlib\ArrayManager;
use Mbe\Shipping\Helper\Data;
use Mbe\Shipping\Model\Ws;
use \Magento\Framework\App\Config\ScopeConfigInterface;

class DataPlugin
{
    protected $shippingHelper;
    protected $ws;
    protected $arrayManager;
    protected $scopeConfig;

    protected $allowedShipmentServicesArray;

    public function __construct(
        Data $shippingHelper,
        Ws  $ws,
        ArrayManager $arrayManager,
        ScopeConfigInterface $scopeConfig
    ) {
        $this->shippingHelper = $shippingHelper;
        $this->ws = $ws;
        $this->arrayManager = $arrayManager;
        $this->scopeConfig = $scopeConfig;

        $this->allowedShipmentServicesArray = $this->shippingHelper->getAllowedShipmentServicesArray();
    }

    public function beforeMerge(\Magento\Config\Model\Config\Structure\Data $subject, array $config)
    {
        $configSection = &$config['config']['system']['sections'];
        if (isset($configSection['carriers']['children'])) {
            $data = $configSection['carriers']['children'];
            if ($this->shippingHelper->isEnabled()) {
                // Free shipping thresholds
                if ($this->arrayManager->exists(Data::MBE_FREE_SHIPMENT_THRESHOLD_GROUP_ID, $data)) {
                    $thresholdFields = $this->getThresholdFields();
                    if (!empty($thresholdFields)) {
                        $data[Data::MBE_FREE_SHIPMENT_THRESHOLD_GROUP_ID]['children'] = $thresholdFields;
                    }
                    $configSection['carriers']['children'] = $data;
                }
                // Custom shipping mapping
                $customMappingFields = $this->getCustomMappingFields();
                if (!empty($customMappingFields)) {
                    $this->shippingHelper->insertIntoArray(
                        $data['mbe_shipping']['children'],
                        $customMappingFields,
                        'enable_custom_mapping'
                    );
                }
                $configSection['carriers']['children'] = $data;

                //Custom label mapping
                $customLabelFields = $this->getCustomLabelFields();
                if (!empty($customLabelFields)) {
                    $this->shippingHelper->insertIntoArray(
                        $data['mbe_shipping']['children'],
                        $customLabelFields,
                        'allowed_shipment_services'
                    );
                }
                $configSection['carriers']['children'] = $data;
            }
        }

        return [$config];
    }

    protected function getThresholdFields()
    {
        $thresholdFields = [];

        foreach ($this->allowedShipmentServicesArray as $index => $allowedShipmentService) {
            $thresholdFields['threshold_'.strtolower($allowedShipmentService).'_dom'] = [
                'id' => strtolower($allowedShipmentService).'_dom',
                'type' => 'text',
                'translate' => 'label',
                'sortOrder' => (($index+1) * 10),
                'showInDefault' => '1',
                'showInWebsite' => '0',
                'showInStore' => '0',
                'label' => __('%1 Domestic', $this->ws->getLabelFromShipmentType($allowedShipmentService) .' - '),
                '_elementType' => 'field',
                'path' => Data::XML_PATH_THRESHOLD
            ];
            $thresholdFields['threshold_'.strtolower($allowedShipmentService).'_ww'] = [
                'id' => strtolower($allowedShipmentService).'_ww',
                'type' => 'text',
                'translate' => 'label',
                'sortOrder' => (($index+1) * 10)+1,
                'showInDefault' => '1',
                'showInWebsite' => '0',
                'showInStore' => '0',
                'label' => __('%1 Rest of the world', $this->ws->getLabelFromShipmentType($allowedShipmentService) .' - '),
                '_elementType' => 'field',
                'path' => Data::XML_PATH_THRESHOLD
            ];
        }

        return $thresholdFields;
    }

    protected function getCustomLabelFields()
    {
        $customLabelFields = [];

        foreach ($this->allowedShipmentServicesArray as $index => $allowedShipmentService) {
            $customLabelFields[Data::MBE_SHIPMENT_CUSTOM_LABEL_PREFIX . strtolower($allowedShipmentService)] = [
                'id' => Data::MBE_SHIPMENT_CUSTOM_LABEL_PREFIX . strtolower($allowedShipmentService),
                'type' => 'text',
                'translate' => 'label',
                'showInDefault' => '1',
                'showInWebsite' => '0',
                'showInStore' => '0',
                'label' => __('Custom label for %1', ' - '. $this->ws->getLabelFromShipmentType($allowedShipmentService)),
                'comment' => __('Insert the custom name for the shipment method. Leave it blank if you don\'t want to change the default value'),
                '_elementType' => 'field',
                'path' => Data::XML_PATH_SHIPMENT
            ];
        }
        return $customLabelFields;
    }

    protected function getCustomMappingFields()
    {
        $defaultMethods = $this->shippingHelper->getDefaultShippingMethods();

        $customMappingFields = [];
        if ($this->shippingHelper->isEnabledCustomMapping()) {
            $selectedOptions['option'][0] = ["label"=>' ',"value"=>''];
            $selectedOptions['option'] = array_merge($selectedOptions['option'], $this->shippingHelper->getShippingCustomMappingOptions());

            foreach ($defaultMethods as $default_method) {
                $methodId = $default_method->getId();
                $methodName = $this->scopeConfig->getValue('carriers/'. $methodId.'/title');

                if (isset($methodId)) {
                    $customMappingFields[Data::MBE_SHIPMENT_CUSTOM_MAPPING_PREFIX . strtolower($methodId)] = [
                        'id' => Data::MBE_SHIPMENT_CUSTOM_MAPPING_PREFIX . strtolower($methodId),
                        'type' => 'select',
                        'translate' => 'label',
                        'showInDefault' => '1',
                        'showInWebsite' => '0',
                        'showInStore' => '0',
                        'label' => __('Custom mapping for %1', (' - '.  $methodName)),
                        'comment' => __('Select the custom mapping for the default shipping method. Leave blank if you don\'t want to map it'),
                        '_elementType' => 'field',
                        'path' => Data::XML_PATH_SHIPMENT,
                        'options' => $selectedOptions
                    ];
                }
            }
            return $customMappingFields;
        }
        return false;
    }
}
