<?php
namespace Mbe\Shipping\Helper;

use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Module\Dir;
use Magento\Framework\Escaper;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Shipping\Model\Config;

class Data extends \Magento\Framework\App\Helper\AbstractHelper
{
    private $_mbeMediaDir = "mbe";
    private $_csvDir = 'csv';
    const MBE_MODULE_FOLDER = 'Mbe/Shipping';
    const MBE_LOG_WS = 'mbe_ws.log';
    const MBE_LOG_PLUGIN = 'mbe_shipping.log';
    const MBE_SHIPPING_PREFIX = "mbe_shipping_";
    const MBE_CSV_PACKAGE_TEMPLATE = 'mbe_package_csv_template.csv';
    const MBE_CSV_PACKAGES_TABLE_NAME = 'mbe_shipping_standard_packages';
    const MBE_CSV_PACKAGE_PRODUCT_TEMPLATE = 'mbe_package_product_csv_template.csv';
    const MBE_CSV_PACKAGES_PRODUCT_TABLE_NAME = 'mbe_shipping_standard_package_product';
    const MBE_CSV_PACKAGES_RESERVED_CODE = 'settings';
    const MBE_CSV_RATES_TABLE_NAME = 'mbeshippingrate';
    const MBE_CSV_RATES_TEMPLATE_CSV = 'mbe_csv_template.csv';
    const XML_PATH_SHIPMENT = 'carriers/mbe_shipping';

    //MAIN
    const XML_PATH_DEBUG = 'carriers/mbe_shipping/debug';
    const XML_PATH_ENABLED = 'carriers/mbe_shipping/active';
    const XML_PATH_COUNTRY = 'carriers/mbe_shipping/country';
    //WS
    const XML_PATH_WS_URL = 'carriers/mbe_shipping/url';
    const XML_PATH_WS_USERNAME = 'carriers/mbe_shipping/username';
    const XML_PATH_WS_PASSWORD = 'carriers/mbe_shipping/password';
    //OPTIONS
    const XML_PATH_DESCRIPTION = 'carriers/mbe_shipping/description';
    const XML_PATH_DEFAULT_SHIPMENT_TYPE = 'carriers/mbe_shipping/default_shipment_type';
    const XML_PATH_ALLOWED_SHIPMENT_SERVICES = 'carriers/mbe_shipping/allowed_shipment_services';
    const XML_PATH_SHIPMENT_CONFIGURATION_MODE = 'carriers/mbe_shipping/shipment_configuration_mode';
    // STANDARD PACKAGES CSV
    const XML_PATH_CSV_STANDARD_PACKAGE_USE_CSV = 'carriers/mbe_shipping/use_packages_csv';
    const XML_PATH_CSV_STANDARD_PACKAGE_DEFAULT = 'carriers/mbe_shipping/packages_default';

    const HANDLING_TYPE_PERCENT = 'P';
    const HANDLING_TYPE_FIXED = 'F';

    const XML_PATH_DEFAULT_LENGTH = 'carriers/mbe_shipping/default_length';
    const XML_PATH_DEFAULT_WIDTH = 'carriers/mbe_shipping/default_width';
    const XML_PATH_WEIGHT_UOM = 'carriers/mbe_shipping/weight_uom';
    const XML_PATH_DEFAULT_HEIGHT = 'carriers/mbe_shipping/default_height';
    const XML_PATH_MAX_PACKAGE_WEIGHT = 'carriers/mbe_shipping/max_package_weight';
    const XML_PATH_MAX_SHIPMENT_WEIGHT = 'carriers/mbe_shipping/max_shipment_weight';
    const XML_PATH_HANDLING_TYPE = 'carriers/mbe_shipping/handling_type';
    const XML_PATH_HANDLING_ACTION = 'carriers/mbe_shipping/handling_action';
    const XML_PATH_HANDLING_FEE = 'carriers/mbe_shipping/handling_fee';
    const XML_PATH_HANDLING_FEE_ROUNDING = 'carriers/mbe_shipping/handling_fee_rounding';
    const XML_PATH_HANDLING_FEE_ROUNDING_AMOUNT = 'carriers/mbe_shipping/handling_fee_rounding_amount';
    const XML_PATH_SALLOWSPECIFIC = 'carriers/mbe_shipping/sallowspecific';
    const XML_PATH_SPECIFICCOUNTRY = 'carriers/mbe_shipping/specificcountry';
    const XML_PATH_SORT_ORDER = 'carriers/mbe_shipping/sort_order';
    const XML_PATH_MAXIMUM_TIME_FOR_SHIPPING_BEFORE_THE_END_OF_THE_DAY = 'carriers/mbe_shipping/maximum_time_for_shipping_before_the_end_of_the_day';
    const XML_PATH_SPECIFICERRMSG = 'carriers/mbe_shipping/specificerrmsg';
    const XML_PATH_WEIGHT_TYPE = 'carriers/mbe_shipping/weight_type';
    const XML_PATH_SHIPMENTS_CLOSURE_MODE = 'carriers/mbe_shipping/shipments_closure_mode';
    const XML_PATH_SHIPMENTS_CLOSURE_TIME = 'carriers/mbe_shipping/shipments_closure_time';
    const XML_PATH_SHIPMENTS_CREATION_MODE = 'carriers/mbe_shipping/shipments_creation_mode';

    const XML_PATH_SHIPMENTS_CSV = 'carriers/mbe_shipping/shipments_csv';
    const XML_PATH_PACKAGES_CSV = 'carriers/mbe_shipping/packages_csv';
    const XML_PATH_PACKAGES_PRODUCT_CSV = 'carriers/mbe_shipping/packages_product_csv';

    const XML_PATH_SHIPMENTS_CSV_MODE = 'carriers/mbe_shipping/shipments_csv_mode';
    const XML_PATH_SHIPMENTS_CSV_INSURANCE_MIN = 'carriers/mbe_shipping/mbe_shipments_csv_insurance_min';
    const XML_PATH_SHIPMENTS_CSV_INSURANCE_PERCENTAGE = 'carriers/mbe_shipping/mbe_shipments_csv_insurance_per';
    const XML_PATH_SHIPMENTS_INSURANCE_MODE = 'carriers/mbe_shipping/mbe_shipments_ins_mode';

    const MBE_FREE_SHIPMENT_THRESHOLD_GROUP_ID = 'mbe_shipping_thresholds';
    const XML_PATH_THRESHOLD = 'carriers/mbe_shipping_thresholds';

    const MBE_SHIPMENT_CUSTOM_LABEL_PREFIX = 'custom_label_';

    const XML_PATH_ENABLE_CUSTOM_MAPPING = 'carriers/mbe_shipping/enable_custom_mapping';
    const MBE_SHIPMENT_CUSTOM_MAPPING_PREFIX = 'custom_mapping_';

    const XML_PATH_SHIP_TO_UAP = 'carriers/mbe_shipping/mbe_ship_to_UAP';

    //const

    const MBE_SHIPMENT_STATUS_CLOSED = "Closed";
    const MBE_SHIPMENT_STATUS_OPEN = "Opened";

    const MBE_CLOSURE_MODE_AUTOMATICALLY = 'automatically';
    const MBE_CLOSURE_MODE_MANUALLY = 'manually';

    const MBE_CREATION_MODE_AUTOMATICALLY = 'automatically';
    const MBE_CREATION_MODE_MANUALLY = 'manually';

    const MBE_CSV_MODE_DISABLED = 'disabled';
    const MBE_CSV_MODE_TOTAL = 'total';
    const MBE_CSV_MODE_PARTIAL = 'partial';
    const MBE_INSURANCE_WITH_TAXES = 'insurance_with_taxes';
    const MBE_INSURANCE_WITHOUT_TAXES = 'insurance_without_taxes';
    const MBE_SHIPPING_WITH_INSURANCE_CODE_SUFFIX = '_INSURANCE';
    const MBE_SHIPPING_WITH_INSURANCE_LABEL_SUFFIX = ' + Insurance';
    const MBE_SHIPPING_TRACKING_SEPARATOR = ',';
    const MBE_SHIPPING_TRACKING_SEPARATOR__OLD = '--';

    const XML_PATH_CURRENT_CSV = 'carriers/mbe_shipping/current_csv';

    const XML_PATH_CACHEDCUSTOMERDATA = 'carriers/mbe_shipping/cachedcustomerdata';

    /**
     * @var \Mbe\Shipping\Model\WsFactory
     */
    protected $shippingWsFactory;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     *
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Sales\Model\Order\ShipmentFactory
     */
    protected $salesOrderShipmentFactory;

    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\CollectionFactory
     */
    protected $salesResourceModelOrderCollectionFactory;

    /**
     *
     * @var \Magento\Sales\Model\Order\Shipment
     */
    protected $shipment;

    /**
     *
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    protected $mediaUrl;

    protected $fileSystem;

    protected $serializer;

    protected $escaper;

    protected $config;

    protected $orderRepository;

    protected $standardPackagesFactory;
    protected $standardPackagesProductFactory;

    protected $moduleDir;

    private $writerConfig;

    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Mbe\Shipping\Model\WsFactory                                       $shippingWsFactory,
        \Magento\Sales\Model\Order\ShipmentFactory                          $salesOrderShipmentFactory,
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory          $salesResourceModelOrderCollectionFactory,
        \Magento\Sales\Model\Order\ShipmentRepository                       $shipment,
        \Magento\Store\Model\StoreManagerInterface                          $storeManager,
        \Magento\Framework\Filesystem                                       $fileSystem,
        \Magento\Framework\Serialize\SerializerInterface                    $serializer,
        OrderRepositoryInterface                                            $orderRepository,
        Escaper                                                             $escaper,
        Config                                                              $config,
        WriterInterface                                                     $writerConfig,
        \Mbe\Shipping\Model\ResourceModel\Packages\CollectionFactory        $standardPackagesFactory,
        \Mbe\Shipping\Model\ResourceModel\PackagesProduct\CollectionFactory $standardPackagesProductFactory,
        Dir $moduleDir

    ) {
        $this->shippingWsFactory = $shippingWsFactory;
        $this->salesOrderShipmentFactory = $salesOrderShipmentFactory;
        $this->salesResourceModelOrderCollectionFactory = $salesResourceModelOrderCollectionFactory;
        $this->shipment = $shipment;
        $this->storeManager = $storeManager;
        $this->mediaUrl = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
        $this->fileSystem = $fileSystem;
        $this->serializer = $serializer;
        $this->escaper = $escaper;
        $this->config = $config;
        $this->orderRepository = $orderRepository;
        $this->standardPackagesFactory = $standardPackagesFactory;
        $this->standardPackagesProductFactory = $standardPackagesProductFactory;
        $this->writerConfig = $writerConfig;
        $this->moduleDir = $moduleDir;
        parent::__construct(
            $context
        );
    }

    public function checkMbeDir()
    {
        try {
            $mediaDir = $this->fileSystem->getDirectoryWrite(DirectoryList::MEDIA);
            if (!$mediaDir->isDirectory($this->_mbeMediaDir)) {
                $mediaDir->create($this->_mbeMediaDir);
            }

            if (!$mediaDir->isDirectory($this->_mbeMediaDir . DIRECTORY_SEPARATOR . $this->_csvDir)) {
                $mediaDir->create($this->_mbeMediaDir . DIRECTORY_SEPARATOR . $this->_csvDir);
            }
        } catch (FileSystemException $e) {
            // TODO: Valutare se tenere tutti i catch o usare uno unico magari con resume
            return false;
        }

        try {
            $appDir = $this->fileSystem->getDirectoryWrite(DirectoryList::APP)->getAbsolutePath();
            $mbeCsvDir = $mediaDir->getAbsolutePath($this->_mbeMediaDir . DIRECTORY_SEPARATOR . $this->_csvDir);
//            $templatePath = $appDir . 'code/' . self::MBE_MODULE_FOLDER . '/Templates/';
            $templatePath = $this->moduleDir->getDir($this->_getModuleName()). '/Templates/';

            // csv rates template
            $destinationPath = $mbeCsvDir . DIRECTORY_SEPARATOR . self::MBE_CSV_RATES_TEMPLATE_CSV;
            if (!$mediaDir->isFile($destinationPath)) {
                $mediaDir->getDriver()->copy(
                    $templatePath . self::MBE_CSV_RATES_TEMPLATE_CSV,
                    $destinationPath
                );
            }

            // csv packages template
            $destinationPath = $mbeCsvDir . DIRECTORY_SEPARATOR . self::MBE_CSV_PACKAGE_TEMPLATE;
            if (!$mediaDir->isFile($destinationPath)) {
                $mediaDir->getDriver()->copy(
                    $templatePath . self::MBE_CSV_PACKAGE_TEMPLATE,
                    $destinationPath
                );
            }

            // csv product packages template
            $destinationPath = $mbeCsvDir . DIRECTORY_SEPARATOR . self::MBE_CSV_PACKAGE_PRODUCT_TEMPLATE;
            if (!$mediaDir->isFile($destinationPath)) {
                $mediaDir->getDriver()->copy(
                    $templatePath . self::MBE_CSV_PACKAGE_PRODUCT_TEMPLATE,
                    $destinationPath
                );
            }

        } catch (FileSystemException $e) {
            // TODO:
        }

    }

    /**
     * @return string
     * @throws FileSystemException
     */
    public function getPluginVersion()
    {
        $appDir = $this->fileSystem->getDirectoryReadByPath($this->moduleDir->getDir($this->_getModuleName()));
        try {
            $composerFile = $appDir->readFile('composer.json');
            $plugin_data = $this->serializer->unserialize($composerFile);
            if (empty($plugin_data['version'])) {
                return '';
            }
            return $plugin_data['version'];
        } catch (FileSystemException $e) {
            throw new FileSystemException(__('Module\'s composer.json file not found'));
        }
    }

    //TODO: check if is necessary to specify storeid or get current storeid
    public function isEnabled($storeId = null)
    {
        $ws = $this->shippingWsFactory->create();
        $result = $ws->isCustomerActive() && $this->scopeConfig->getValue(self::XML_PATH_ENABLED, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
        return $result;
    }

    public function debug($storeId = null)
    {
        $result = ($this->scopeConfig->getValue(self::XML_PATH_DEBUG, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId) == 1);
        return $result;
    }

    public function getCountry($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_COUNTRY, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }


    public function getWsUrl($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_WS_URL, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getWsUsername($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_WS_USERNAME, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getWsPassword($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_WS_PASSWORD, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getDescription($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_DESCRIPTION, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getDefaultShipmentType($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_DEFAULT_SHIPMENT_TYPE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getAllowedShipmentServices($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_ALLOWED_SHIPMENT_SERVICES, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShipToUap($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIP_TO_UAP, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function convertShippingCodeWithInsurance($code)
    {
        return $code . self::MBE_SHIPPING_WITH_INSURANCE_CODE_SUFFIX;
    }

    public function convertShippingLabelWithInsurance($label)
    {
        return $label . self::MBE_SHIPPING_WITH_INSURANCE_LABEL_SUFFIX;
    }

    public function convertShippingCodeWithoutInsurance($code)
    {
        return str_replace(self::MBE_SHIPPING_WITH_INSURANCE_CODE_SUFFIX, "", $code);
    }

    public function isShippingWithInsurance($code)
    {
        $result = false;
        /*
        $shippingSuffix = substr($code, -strlen(self::MBE_SHIPPING_WITH_INSURANCE_CODE_SUFFIX));
        if ($shippingSuffix == self::MBE_SHIPPING_WITH_INSURANCE_CODE_SUFFIX) {
            $result = true;
        }
        */
        if (strpos($code, self::MBE_SHIPPING_WITH_INSURANCE_CODE_SUFFIX) !== false) {
            $result = true;
        }
        return $result;
    }

    public function getAllowedShipmentServicesArray($storeId = null)
    {
        $allowedShipmentServices = $this->getAllowedShipmentServices($storeId);

        $result = array();
        $allowedShipmentServicesArray = array();

        if ($allowedShipmentServices != "") {
            $allowedShipmentServicesArray = explode(",", $allowedShipmentServices);
        }

        /** @var $ws Mbe_Shipping_Model_Ws */
        $ws = $this->shippingWsFactory->create();
        $canSpecifyInsurance = $ws->getCustomerPermission('canSpecifyInsurance');

        foreach ($allowedShipmentServicesArray as $item) {
            $canAdd = true;
            if (!$canSpecifyInsurance) {
                if (strpos($item, self::MBE_SHIPPING_WITH_INSURANCE_CODE_SUFFIX) !== false) {
                    $canAdd = false;
                }
            }
            if ($canAdd) {
                array_push($result, $item);
            }
        }

        return $result;
    }

    public function getShipmentConfigurationMode($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENT_CONFIGURATION_MODE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getDefaultLength($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_DEFAULT_LENGTH, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getDefaultWidth($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_DEFAULT_WIDTH, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getDefaultHeight($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_DEFAULT_HEIGHT, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    /**
     * Check the max weight and replace it if necessary
     *
     * @param $baseWeight
     * @param $type
     * @return float|mixed|null
     */
    protected function checkMaxWeight($baseWeight, $type)
    {
        $ws = $this->shippingWsFactory->create();
        if ($this->getDefaultShipmentType() == "ENVELOPE") {
            $maxWeight = $this->getWeight(
                0.5,
                $this->scopeConfig->getValue(\Magento\Directory\Helper\Data::XML_PATH_WEIGHT_UNIT),
                'kgs'
            );
        } else {
            $maxWeight = $this->getWeight(
                $ws->getCustomerPermission($type),
                $this->scopeConfig->getValue(\Magento\Directory\Helper\Data::XML_PATH_WEIGHT_UNIT),
                'kgs'
            );
        }
        if ($maxWeight > 0 && $maxWeight < $baseWeight) {
            $baseWeight = $maxWeight;
        }
        return $baseWeight;
    }

    public function getMaxPackageWeight($storeId = null)
    {
        $result = $this->scopeConfig->getValue(self::XML_PATH_MAX_PACKAGE_WEIGHT, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
        return $this->checkMaxWeight($result, "maxParcelWeight");
    }

    public function getMaxShipmentWeight($storeId = null)
    {
        $configMaxShipmentWeight = $this->scopeConfig->getValue(self::XML_PATH_MAX_SHIPMENT_WEIGHT, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
        return $this->checkMaxWeight($configMaxShipmentWeight, "maxShipmentWeight");
    }

    public function getHandlingType($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_HANDLING_TYPE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getHandlingAction($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_HANDLING_ACTION, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getHandlingFee($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_HANDLING_FEE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getHandlingFeeRounding($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_HANDLING_FEE_ROUNDING, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getHandlingFeeRoundingAmount($storeId = null)
    {
        $result = 1;
        if ($this->scopeConfig->getValue(self::XML_PATH_HANDLING_FEE_ROUNDING_AMOUNT, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId) == 2) {
            $result = 0.5;
        }
        return $result;
    }

    public function getSallowspecific($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SALLOWSPECIFIC, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getSpecificcountry($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SPECIFICCOUNTRY, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getSortOrder($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SORT_ORDER, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getMaximumTimeForShippingBeforeTheEndOfTheDay($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_MAXIMUM_TIME_FOR_SHIPPING_BEFORE_THE_END_OF_THE_DAY, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getSpecificerrmsg($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SPECIFICERRMSG, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getWeightType($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_WEIGHT_TYPE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShipmentsClosureMode($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENTS_CLOSURE_MODE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShipmentsClosureTime($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENTS_CLOSURE_TIME, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShipmentsCreationMode($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENTS_CREATION_MODE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }


    public function round($value)
    {
        $result = $value;
        $handlingFeeRounding = $this->getHandlingFeeRounding();
        $handlingFeeRoundingAmount = $this->getHandlingFeeRoundingAmount();

        if ($handlingFeeRounding == 2) {

            if ($handlingFeeRoundingAmount == 1) {
                $result = round($value, 0);
            }
            else {
                $result = round($value, 2);
            }

        }
        elseif ($handlingFeeRounding == 3) {
            if ($handlingFeeRoundingAmount == 1) {
                $result = floor($value);
            }
            else {
                $result = floor($value * 2) / 2;
            }
        }
        elseif ($handlingFeeRounding == 4) {
            if ($handlingFeeRoundingAmount == 1) {
                $result = ceil($value);
            }
            else {
                $result = ceil($value * 2) / 2;
            }
        }
        return $result;
    }


    public function testRound($value)
    {
        $valueRounded = $this->round($value);
    }

    public function getNameFromLabel($label)
    {
        $name = $label;
        $name = strtolower($name);
        $name = str_replace(" ", "_", $name);
        $name = str_replace(" ", "_", $name);
        return $name;
    }

    public function getThresholdByShippingServrice($shippingService, $storeId = null)
    {
        $shippingService = strtolower($shippingService);
        return $this->escaper->escapeHtml($this->scopeConfig->getValue(
            self::XML_PATH_THRESHOLD . "/" . $shippingService,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $storeId
        ));
    }


    public function getTrackingStatus($trackingNumber)
    {
        $result = self::MBE_SHIPMENT_STATUS_OPEN;
        $mediaDir = $this->fileSystem->getDirectoryRead(DirectoryList::MEDIA);

        if ($mediaDir->isFile($this->_mbeMediaDir . DIRECTORY_SEPARATOR . $trackingNumber . ".pdf")) {
            $result = self::MBE_SHIPMENT_STATUS_CLOSED;
        }
        return $result;
    }

    public function isShippingOpen($shipmentId)
    {
        $result = true;
        $shipment = $this->shipment->get($shipmentId);
        $tracks = $shipment->getAllTracks();
        foreach ($tracks as $track) {
            $trackingNumber = $track->getTrackNumber();
            $result = $result && $this->isTrackingOpen($trackingNumber);
        }
        return $result;
    }

    public function isTrackingOpen($trackingNumber)
    {
        return $this->getTrackingStatus($trackingNumber) == self::MBE_SHIPMENT_STATUS_OPEN;
    }


    public function getShipmentFilePath($shipmentIncrementId, $ext)
    {
        return $this->fileSystem
            ->getDirectoryRead(DirectoryList::MEDIA)
            ->getAbsolutePath($this->_mbeMediaDir . DIRECTORY_SEPARATOR . $shipmentIncrementId . "." . $ext);
    }

//    public function getShipmentUrlByShipmentId($shipmentId)
//    {
//        $shipment = $this->salesOrderShipmentFactory->create()->load($shipmentId);
//        return $this->getShipmentUrl($shipment->getIncrementId());
//    }

    public function getShipmentUrl($shipmentIncrementId)
    {
        $this->checkMbeDir();
        $mbeFileUrl = $this->mediaUrl . $this->_mbeMediaDir . DIRECTORY_SEPARATOR . $shipmentIncrementId;
        $filePath = $this->_mbeMediaDir . DIRECTORY_SEPARATOR . $shipmentIncrementId;
        $mediaDir = $this->fileSystem->getDirectoryRead(DirectoryList::MEDIA);

        if ($mediaDir->isFile($filePath . ".pdf")) {
            $result = $mbeFileUrl . ".pdf";
        } elseif ($mediaDir->isFile($filePath . ".html")) {
            $result = $mbeFileUrl . ".html";
        } elseif ($mediaDir->isFile($filePath . ".gif")) {
            $result = $mbeFileUrl . ".gif";
        } else {
            $foundFiles = true;
            $i = 1;
            $resultFiles = [];
            do {
                if ($mediaDir->isFile($filePath . '_' . $i . ".pdf")) {
                    $resultFiles[] = $mbeFileUrl . '_' . $i . ".pdf";
                } elseif ($mediaDir->isFile($filePath . '_' . $i . ".html")) {
                    $resultFiles[] = $mbeFileUrl . '_' . $i . ".html";
                } elseif ($mediaDir->isFile($filePath . '_' . $i . ".gif")) {
                    $resultFiles[] = $mbeFileUrl . '_' . $i . ".gif";
                } else {
                    $foundFiles = false;
                }
                $i++;

            } while ($foundFiles);

            $result = $resultFiles;
        }

        return $result;
    }

    public function getTrackingFilePath($trackingNumber)
    {
        $this->checkMbeDir();
        $mediaDir = $this->fileSystem->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath();
        $mbeDir = $mediaDir . DIRECTORY_SEPARATOR . $this->_mbeMediaDir;
        return $mbeDir . DIRECTORY_SEPARATOR . $trackingNumber . ".pdf";
    }

    public function getTrackingUrlByShipmentId($shipmentId)
    {
        $shipment = $this->shipment->get($shipmentId);
        $tracks = $shipment->getAllTracks();
        foreach ($tracks as $track) {
            $trackingNumber = $track->getTrackNumber();
            return $this->getTrackingUrl($trackingNumber);
        }
    }

    public function getTrackingUrl($trackingNumber)
    {
        $mbeUrl = $this->mediaUrl . $this->_mbeMediaDir . '/';
        $fileUrl = $mbeUrl . $trackingNumber . ".pdf";
        return $fileUrl;
    }

    public function getTrackings($shipmentId)
    {
        $tracking = get_post_meta($shipmentId, self::SHIPMENT_SOURCE_TRACKING_NUMBER, true);
        if (strpos($tracking, self::MBE_SHIPPING_TRACKING_SEPARATOR) !== false) {
            $value = explode(self::MBE_SHIPPING_TRACKING_SEPARATOR, $tracking);

        }
        else {
            $value = explode(self::MBE_SHIPPING_TRACKING_SEPARATOR__OLD, $tracking);
        }

        return is_array($value) ? (array_filter($value, function ($value) {
            return $value !== '';
        })) : $value;
    }

    public function getTrackingsString($shipmentId)
    {
        $result = get_post_meta($shipmentId, self::SHIPMENT_SOURCE_TRACKING_NUMBER, true);
        //compatibility replace
        if (strpos($result, self::MBE_SHIPPING_TRACKING_SEPARATOR__OLD) !== false) {
            $result = str_replace(self::MBE_SHIPPING_TRACKING_SEPARATOR__OLD, self::MBE_SHIPPING_TRACKING_SEPARATOR, $result);
        }
        return $result;
    }


    public function getShippingTypesFromShipments()
    {
        $result = array();

        $ws = $this->shippingWsFactory->create();
        $collection = $this->salesResourceModelOrderCollectionFactory->create()->addFieldToSelect('shipping_method');

        $collection->getSelect()->group('shipping_method');

        foreach ($collection as $col) {
            $shippingMethodCode = $this->getShippingCodeFromMagentoShippingMethod($col->getShippingMethod());
            $shippingMethodLabel = $ws->getLabelFromShipmentType($shippingMethodCode);

            if ($this->isShippingWithInsurance($col->getShippingMethod())) {
                $shippingMethodLabel = $this->convertShippingLabelWithInsurance($shippingMethodLabel);
            }
            $result[$col->getShippingMethod()] = $shippingMethodLabel;
        }
        return $result;
    }


    public function getShipmentLabel($shipmentCode)
    {
        /** @var $ws Mbe_Shipping_Model_Ws */
        $ws = $this->shippingWsFactory->create();

        return $ws->getLabelFromShipmentType($shipmentCode);
    }

    public function getShipmentLabelFromMagentoShippingMethod($shipmentCode)
    {
        $shippingMethod = $this->getShippingCodeFromMagentoShippingMethod($shipmentCode);
        $result = $this->getShipmentLabel($shippingMethod);
        if ($this->isShippingWithInsurance($shipmentCode)) {
            $result = $this->convertShippingLabelWithInsurance($result);
        }
        return $result;
    }

    public function isMbeShippingCustomMapping($shippingCarrier)
    {
        // double check it's not a mbe shipping method and custom mapping is enabled
        // Due to Magento bug , the carrier code is retrieved as 'mbe' instead of 'mbe_shipping'
        if ($shippingCarrier !== 'mbe' && $this->isEnabledCustomMapping()) {
            $customMapping = $this->getCustomMappingShippingMethods();
            if (isset($customMapping[$shippingCarrier]) && !empty($customMapping[$shippingCarrier])) {
                return true;
            }
        }
        return false;
    }

    public function isMbeShipping($order)
    {
        $shippingMethod = $order ? $order->getShippingMethod() : null;
        // Check if the methods is an MBE one or if is a custom mapped default one
        if (strpos($shippingMethod, self::MBE_SHIPPING_PREFIX) !== false
            || $this->isMbeShippingCustomMapping($order->getShippingMethod(true)->getCarrierCode())
        ){
            return true;
        }
        return false;
    }

    /**
     * @param $order \Magento\Sales\Model\Order
     */
    public function getShippingMethod($order)
    {
        $order_item_id = $order->getId();

        if ($order_item_id) {
            $shippingCarrier = $order->getShippingMethod(true)->getCarrierCode();
            $shippingMethod = $order->getShippingMethod();
            $customMapping = $this->getShippingCustomMapping($shippingCarrier);
            if (!empty($customMapping)) {
                $shippingMethod = $customMapping;
            }
            return $shippingMethod;
        } else {
            return false;
        }
    }

    public function getShippingCustomMappingOptions()
    {
        /** @var $ws Mbe_Shipping_Model_Ws */
        $ws = $this->shippingWsFactory->create();

        $availableShipping = $ws->getAllowedShipmentServices();
        $selectedOptions = [];
        foreach ($this->getAllowedShipmentServicesArray() as $key => $value) {
            $index = array_search($value, array_column($availableShipping, 'value'));
            $label = '';
            if (isset($availableShipping[$index]['label'])) {
                $label = __($availableShipping[$index]['label'])->getText();
            }
            $selectedOptions[] = [
                "label" => $label,
                "value" => $value
            ];
        }
        return $selectedOptions;
    }

    public function getCustomMappingShippingMethods()
    {
        $defaultMethods = $this->getDefaultShippingMethods();
        $customMapping = [];
        foreach ($defaultMethods as $default_method) {
            $mapping = $this->getShippingCustomMapping(strtolower($default_method->getId()));
            if (!empty($mapping)) {
                $customMapping[$default_method->getId()] = strtolower($mapping);
            }
        }
        return $customMapping;
    }

    public function getDefaultShippingMethods()
    {
        $filterMethods = ['flatrate'=>'', 'freeshipping'=>'', 'tablerate'=>''];
        return array_intersect_key($this->config->getActiveCarriers(), $filterMethods);
    }

    public function isEnabledCustomMapping($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_ENABLE_CUSTOM_MAPPING, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShippingCustomMapping($shippingCarrier)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENT .'/'. self::MBE_SHIPMENT_CUSTOM_MAPPING_PREFIX .$shippingCarrier);
    }


    public function setOrderCustomMappingInfo($order)
    {
        try {
            // Add custom mapping flag for shipment list and comment to track it
            if ($this->isMbeShippingCustomMapping($order->getShippingMethod(true)->getCarrierCode())
                && $order
            ) {
                if (!$order->getIsMbeCustomMapping()) { // check if the flag and comment are alredy set
                    $ws = $this->shippingWsFactory->create();
                    $order->setIsMbeCustomMapping(true); // set the custom mapping flag
                    $carrierCode = $order->getShippingMethod(true)->getCarrierCode();
                    $shippingMbeMethod = $this->getShippingCustomMapping($carrierCode);

                    $availableShipping = $ws->getAllowedShipmentServices();
                    $index = array_search($shippingMbeMethod, array_column($availableShipping, 'value'));
                    if (isset($availableShipping[$index]['label'])) {
                        $customMappingDescription = __($availableShipping[$index]['label'])->getText();
                    }

                    $order->addCommentToStatusHistory(
                        __('Custom mapping for %1', $this->scopeConfig->getValue('carriers/' . $carrierCode . '/title')) .
                        ' - ' .
                        $customMappingDescription
                    );

                    $order->save();
                }
                return true;
            }
        } catch (NoSuchEntityException|InputException $e) {
            return false;
        }
        return false;
    }
//    public function getMbeShippingType($order)
//    {
//        $result = "";
//        $shippingMethod = $order->getShippingMethod();
//        if (strpos($shippingMethod, self::MBE_SHIPPING_PREFIX) !== false) {
//
//            $result = $this->getShippingCodeFromMagentoShippingMethod($shippingMethod);
//        }
//        return $result;
//    }

    public function getShippingParamFromMagentoShippingMethod($shippingMethod, $i)
    {
        $shippingMethodWithSubZone = str_replace(self::MBE_SHIPPING_PREFIX, '', $shippingMethod);

        $shippingMethodWithSubZoneArray = explode('_', $shippingMethodWithSubZone);
        if (isset($shippingMethodWithSubZoneArray[$i])) {
            return $shippingMethodWithSubZoneArray[$i];
        }
        return null;
    }

    public function getShippingCodeFromMagentoShippingMethod($shippingMethod)
    {
        return $this->getShippingParamFromMagentoShippingMethod($shippingMethod, 0);
    }

    public function getShippingSubZoneFromMagentoShippingMethod($shippingMethod)
    {
        return $this->getShippingParamFromMagentoShippingMethod($shippingMethod, 1);
    }

    public function getShipmentsCsv($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENTS_CSV, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShipmentsCsvMode($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENTS_CSV_MODE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShipmentsCsvInsuranceMin($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENTS_CSV_INSURANCE_MIN, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShipmentsCsvInsurancePercentage($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENTS_CSV_INSURANCE_PERCENTAGE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getShipmentsInsuranceMode($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_SHIPMENTS_INSURANCE_MODE, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function mbeUploadDir()
    {
        $this->checkMbeDir();
        $mediaDir = $this->fileSystem->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath();
        return $mediaDir . DIRECTORY_SEPARATOR . $this->_mbeMediaDir;
    }

    public function mbeCsvUploadDir()
    {
        $result = $this->mbeUploadDir() . DIRECTORY_SEPARATOR . $this->_csvDir;
        return $result;
    }

    public function getCurrentCsvUrl()
    {
        $fileName = $this->getShipmentsCsv();
        $result = $this->mediaUrl . $this->_mbeMediaDir . '/' . $this->_csvDir . '/' . $fileName;
        return $result;
    }

    public function getCsvTemplateUrl()
    {
        $result = $this->mediaUrl . $this->_mbeMediaDir . '/' . $this->_csvDir . '/' . self::MBE_CSV_RATES_TEMPLATE_CSV;
        return $result;
    }

    public function getLogWsPath()
    {
        return $this->fileSystem->getDirectoryRead(DirectoryList::LOG)
                ->getAbsolutePath(self::MBE_LOG_WS);
    }

    public function getLogPluginPath($absolute = true)
    {
        if($absolute) {
            return $this->fileSystem->getDirectoryRead(DirectoryList::LOG)->getAbsolutePath(self::MBE_LOG_PLUGIN);
        }
        return $this->fileSystem->getDirectoryRead(DirectoryList::LOG)->getRelativePath(self::MBE_LOG_PLUGIN);

    }

    public function getProductTitleFromOrderItem($item)
    {
        $title = $item->getName();

        if ($item->getProductType() == "configurable") {
            $options = $item->getProductOptions();

            if (isset($options['simple_name'])) {
                $title .= ' - ' . $options['simple_name'];
            }
            $variationsString = '';
            if (isset($options['attributes_info'])) {
                foreach ($options['attributes_info'] as $attributesInfo) {
                    if ($variationsString != '') {
                        $variationsString .= ', ';
                    }

                    $variationsString .= $attributesInfo['label'] . ': ' . $attributesInfo['value'];
                }
            }

            if ($variationsString) {
                $title .= ' (' . $variationsString . ')';
            }
        }

        return $title;
    }

    public function convertWeight($weight, $toUnit = 'kgs')
    {
        if (is_array($weight)) {
            foreach ($weight as $key => $value) {
                $weight[$key] = self::getWeight($value, $toUnit);
            }
            return $weight;
        } else {
            return self::getWeight($weight, $toUnit);
        }
    }

    /**
     * Normalise weights, unify to kg then convert to wanted unit value.
     *
     * Usage:
     * wc_get_weight(55, 'kgs');
     * wc_get_weight(55, 'kgs', 'lbs');
     *
     * @param int|float $weight    Weight.
     * @param string    $to_unit   Unit to convert to.
     *                             Options: 'g', 'kgs', 'lbs', 'oz'.
     * @param string    $from_unit Unit to convert from.
     *                             Defaults to ''.
     *                             Options: 'g', 'kgs', 'lbs', 'oz'.
     * @return float
     */
    public function getWeight($weight, $to_unit, $from_unit = '')
    {
        $weight  = (float) $weight;
        $to_unit = strtolower($to_unit);
        $from_unit = strtolower($from_unit);

        if (empty($from_unit)) {
            $from_unit = strtolower($this->scopeConfig->getValue(\Magento\Directory\Helper\Data::XML_PATH_WEIGHT_UNIT));
        }

        // Unify all units to kg first.
        if ($from_unit !== $to_unit) {
            switch ($from_unit) {
                case 'g':
                    $weight *= 0.001;
                    break;
                case 'lbs':
                    $weight *= 0.453592;
                    break;
                case 'oz':
                    $weight *= 0.0283495;
                    break;
            }

            // Output desired unit.
            switch ($to_unit) {
                case 'g':
                    $weight *= 1000;
                    break;
                case 'lbs':
                    $weight *= 2.20462;
                    break;
                case 'oz':
                    $weight *= 35.274;
                    break;
            }
        }

        return ($weight < 0) ? 0 : $weight;
    }

    public function getShippingMethodCustomLabel($methodCode)
    {
        $customLabel = trim($this->scopeConfig->getValue(
            self::XML_PATH_SHIPMENT . '/' . self::MBE_SHIPMENT_CUSTOM_LABEL_PREFIX . strtolower($methodCode)
        ));
        if (!empty($customLabel)) {
            return $this->escaper->escapeHtml($customLabel);
        }
        return false;
    }

    public function getPackagesCsv($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_PACKAGES_CSV, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getCurrentCsvPackagesUrl()
    {
        $packagesCsv = $this->getPackagesCsv();
        if (empty($packagesCsv)) {
            return '';
        }
        return $this->mediaUrl . $this->_mbeMediaDir . '/' . $this->_csvDir . '/' . $packagesCsv;
    }

    public function getCsvTemplatePackagesUrl()
    {
        return $this->mediaUrl . $this->_mbeMediaDir . '/' . $this->_csvDir . '/' . self::MBE_CSV_PACKAGE_TEMPLATE;
    }

    public function getCsvPackagesTableName()
    {
        return self::MBE_CSV_PACKAGES_TABLE_NAME;
    }

    public function getPackagesProductCsv($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_PACKAGES_PRODUCT_CSV, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function getCurrentCsvPackagesProductUrl()
    {
        $packagesProductCsv = $this->getPackagesProductCsv();
        if (empty($packagesProductCsv)) {
            return '';
        }
        return $this->mediaUrl . $this->_mbeMediaDir . '/' . $this->_csvDir . '/' . $packagesProductCsv;
    }

    public function getCsvTemplatePackagesProductUrl()
    {
        return $this->mediaUrl . $this->_mbeMediaDir . '/' . $this->_csvDir . '/' . self::MBE_CSV_PACKAGE_PRODUCT_TEMPLATE;
    }

    public function getCsvPackagesProductTableName()
    {
        return self::MBE_CSV_PACKAGES_PRODUCT_TABLE_NAME;
    }

    public function isCsvStandardPackageEnabled($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_CSV_STANDARD_PACKAGE_USE_CSV, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function disableCsvStandardPackage()
    {
        $this->writerConfig->save(self::XML_PATH_CSV_STANDARD_PACKAGE_USE_CSV, false);
    }

    public function useCsvStandardPackages()
    {
        return $this->isCsvStandardPackageEnabled()
            && $this->standardPackagesFactory->create()->count() > 0
            && \Mbe\Shipping\Model\Carrier::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_SINGLE_PARCEL == $this->getShipmentConfigurationMode();
    }

    public function getCsvStandardPackageDefault($storeId = null)
    {
        return $this->scopeConfig->getValue(self::XML_PATH_CSV_STANDARD_PACKAGE_DEFAULT, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function setCsvStandardPackageDefault($value)
    {
        $this->writerConfig->save(self::XML_PATH_CSV_STANDARD_PACKAGE_DEFAULT, $value);
    }

    /**
     * Return the CSV package information related to a specific product, if any
     * Otherwise it returns the CSV standard package selected as the default one
     *
     * @param null $productSku
     * @return array|null
     *
     */
    protected function getCsvPackageInfo($productSku)
    {
        $packagesInfo = $this->standardPackagesFactory->create()->getPackageInfobyProduct($productSku);
        if ($packagesInfo->count() <= 0) {
            $packagesInfo = $this->standardPackagesFactory
                ->create()
                ->getPackageInfobyId($this->getCsvStandardPackageDefault());
        }
        $packageInfoResult = $packagesInfo->getFirstItem();
        $packageInfoResult->setData(
            'max_weight',
            $this->checkMaxWeight($packageInfoResult->getData('max_weight'), "maxParcelWeight")
        );
        return $packageInfoResult->getData();
    }

    protected function getSettingsPackageInfo($singleParcel)
    {
        return [
            'id' => null,
            'package_code' => self::MBE_CSV_PACKAGES_RESERVED_CODE,
            'package_label' => 'Package from settings',
            'height' => $this->getDefaultHeight(),
            'width' => $this->getDefaultWidth(),
            'length' => $this->getDefaultLength(),
            'max_weight' => $this->getMaxPackageWeight(),
            'single_parcel' => $singleParcel,
            'custom_package' => false
        ];
    }

    /**
     * @param $productSku
     * @param false $singleParcel // All the packages based on settings (not CSV) must be set as "single parcels"
     * @return array|null
     */
    public function getPackageInfo($productSku, $singleParcel = false)
    {
        if ($this->useCsvStandardPackages()) {
            return $this->getCsvPackageInfo($productSku);
        } else {
            return $this->getSettingsPackageInfo($singleParcel);
        }
    }

    /**
     * Insert the elements of an array into an existing one after a specified keys (if found) or append it to the end
     * @param array $arrayInto Array to which the data must be added
     * @param array $arrayFrom Array of data to be added
     * @param null $afterKey Key to search in the main array
     */
    public function insertIntoArray(&$arrayInto, $arrayFrom, $afterKey = null)
    {
        if ($afterKey) {
            $pos = array_search($afterKey, array_keys($arrayInto));
            if ($pos !== false) {
                $arrayInto = array_merge(
                    array_slice($arrayInto, 0, $pos+1, true),
                    $arrayFrom,
                    array_slice($arrayInto, $pos+1, null, true)
                );
            }
        }
        array_merge($arrayInto, $arrayFrom);
    }

    public function getBoxesArray(&$boxesArray, &$boxesSingleParcelArray, $itemWeight, $packageInfo)
    {
        if ($packageInfo['single_parcel'] || $packageInfo['custom_package']) {
            if (!isset($boxesSingleParcelArray[$packageInfo['package_code']])) {
                $boxesSingleParcelArray[$packageInfo['package_code']] = $this->addEmptyBoxType($packageInfo);
            }
            $boxesSingleParcelArray[$packageInfo['package_code']]['weight'][] = $itemWeight;
        } else {
            $canAddToExistingBox = false;
            if (!isset($boxesArray[$packageInfo['package_code']])) {
                $boxesArray[$packageInfo['package_code']] = $this->addEmptyBoxType($packageInfo);
            }
            $boxesPackage = &$boxesArray[$packageInfo['package_code']]; // by ref to simplify the code
            $boxesCount = count($boxesPackage['weight']);
            for ($j = 0; $j < $boxesCount; $j++) {
                $newWeight = $boxesPackage['weight'][$j] + $itemWeight;
                if ($newWeight <= $packageInfo['max_weight']) {
                    $canAddToExistingBox = true;
                    $boxesPackage['weight'][$j] = $newWeight;
                    break;
                }
            }
            if (!$canAddToExistingBox) {
                $boxesPackage['weight'][] = $itemWeight;
            }
        }
        return $boxesArray;
    }

    public function addEmptyBoxType($packageInfo)
    {
        return [
            'maxweight' => $packageInfo['max_weight'],
            'dimensions' => [
                'length' => $packageInfo['length'],
                'width' => $packageInfo['width'],
                'height' => $packageInfo['height']
            ],
            'weight' => []
        ];
    }

    public function mergeBoxesArray($boxes, $boxesSingleParcel)
    {
        foreach ($boxes as $key => $value) {
            if (isset($boxesSingleParcel[$key])) {
                foreach ($boxesSingleParcel[$key]['weight'] as $item) {
                    $boxes[$key]['weight'][] = $item;
                }
                // remove the merged package
                unset($boxesSingleParcel[$key]);
            }
        }
        //  append all the remaining packages
        return array_merge($boxes, $boxesSingleParcel);
    }

    public function countBoxesArray($boxesArray)
    {
        $count = 0;
        $countArray = array_column($boxesArray, 'weight');
        foreach ($countArray as $box) {
            $count += count($box);
        }
        return $count;
    }

    public function getTotalWeight($boxesArray)
    {
        $totalWeight = 0 ;
        foreach ($boxesArray as $box) {
            foreach ($box['weight'] as $weight) {
                $totalWeight += $weight;
            }
        }
        return $totalWeight;
    }

    public function getSubtotalForInsurance($item)
    {
        if ($this->getShipmentsInsuranceMode() === self::MBE_INSURANCE_WITH_TAXES) {
            return $item->getRowTotalInclTax();
        } else {
            return $item->getRowTotal();
        }
    }

    public function isOnlineMBE()
    {
        return (strpos(strtolower($this->getWsUrl()),'onlinembe') !== false);
    }

    /**
     * @param \Magento\Sales\Api\Data\ShipmentTrackInterface[] $trackList
     * @return bool
     */
    public function isReturned($trackList)
    {
        foreach ($trackList as $track) {
            if (strpos(strtolower($track->getData('title')), 'mbe return shipping') !== false) {
                return true;
            }
        }
        return false;
    }
}