<?php

namespace Mbe\Shipping\Model;

use Magento\Quote\Model\Quote\Address\RateResult\Method;
use Magento\Quote\Model\Quote\Item;
use Magento\Shipping\Model\Carrier\AbstractCarrier;
use Magento\Shipping\Model\Carrier\CarrierInterface;
use Mbe\Shipping\Helper\Data;

class Carrier extends AbstractCarrier implements CarrierInterface
{

    const SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_ITEM = 1;
    const SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_SINGLE_PARCEL = 2;
    const SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_MULTI_PARCEL = 3;

    const HANDLING_TYPE_PER_SHIPMENT = "S";
    const HANDLING_TYPE_PER_PARCEL = "P";


    /**
     * Carrier's code, as defined in parent class
     *
     * @var string
     */
    protected $_code = 'mbe_shipping';

    protected $logger = null;
    protected $helper = null;

    /**
     * @var \Mbe\Shipping\Helper\Data
     */
    protected $shippingHelper;

    /**
     * @var \Mbe\Shipping\Helper\Logger
     */
    protected $shippingLoggerHelper;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Shipping\Model\Rate\ResultFactory
     */
    protected $shippingRateResultFactory;

    /**
     * @var \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory
     */
    protected $quoteQuoteAddressRateResultErrorFactory;

    /**
     * @var \Mbe\Shipping\Model\WsFactory
     */
    protected $shippingWsFactory;

    /**
     * @var \Mbe\Shipping\Helper\Rates
     */
    protected $shippingRatesHelper;

    /**
     * @var \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory
     */
    protected $quoteQuoteAddressRateResultMethodFactory;

    /**
     * @var \Mbe\Shipping\Helper\Tracking
     */
    protected $shippingTrackingHelper;

    /**
     * @var \Magento\Shipping\Model\Tracking\Result\StatusFactory
     */
    protected $shippingTrackingResultStatusFactory;

    public function __construct(
        \Mbe\Shipping\Helper\Data $shippingHelper,
        \Mbe\Shipping\Helper\Logger $shippingLoggerHelper,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Shipping\Model\Rate\ResultFactory $shippingRateResultFactory,
        \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory $quoteQuoteAddressRateResultErrorFactory,
        \Mbe\Shipping\Model\WsFactory $shippingWsFactory,
        \Mbe\Shipping\Helper\Rates $shippingRatesHelper,
        \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory $quoteQuoteAddressRateResultMethodFactory,
        \Mbe\Shipping\Helper\Tracking $shippingTrackingHelper,
        \Magento\Shipping\Model\Tracking\Result\StatusFactory $shippingTrackingResultStatusFactory
    ) {
        $this->shippingHelper = $shippingHelper;
        $this->shippingLoggerHelper = $shippingLoggerHelper;
        $this->scopeConfig = $scopeConfig;
        $this->shippingRateResultFactory = $shippingRateResultFactory;
        $this->quoteQuoteAddressRateResultErrorFactory = $quoteQuoteAddressRateResultErrorFactory;
        $this->shippingWsFactory = $shippingWsFactory;
        $this->shippingRatesHelper = $shippingRatesHelper;
        $this->quoteQuoteAddressRateResultMethodFactory = $quoteQuoteAddressRateResultMethodFactory;
        $this->shippingTrackingHelper = $shippingTrackingHelper;
        $this->shippingTrackingResultStatusFactory = $shippingTrackingResultStatusFactory;
    }

    public function getConfigData($field)
    {
        if (empty($this->_code)){
            return false;
        }

        $path = 'carriers/' . $this->_code . '/' . $field;

        return $this->scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $this->getStore());
    }

    public function getRequestWeight(\Magento\Quote\Model\Quote\Address\RateRequest $request)
    {
        /*
        $result = 0;
        foreach ($request->getAllItems() as $item) {
            $result += $item->getWeight();
        }
        return $result;
        */
        return $request->getPackageWeight();
    }

    /**
     * Returns available shipping rates
     *
     * @param \Magento\Quote\Model\Quote\Address\RateRequest $request
     * @return \Magento\Shipping\Model\Rate\Result
     */
    public function collectRates(\Magento\Quote\Model\Quote\Address\RateRequest $request)
    {

        $this->shippingLoggerHelper->log('collectRates');

        /** @var \Magento\Shipping\Model\Rate\Result $result */
        $result = $this->shippingRateResultFactory->create();

        //TODO: check countries

        if ($this->shippingHelper->isEnabledCustomMapping()) {
            // If custom mapping is used no MBE methods will be available
            $this->shippingLoggerHelper->log('custom mapping enabled');
            return $result;
        } else {
            if (!$this->shippingHelper->isEnabled($this->getStoreId())) {
                $this->shippingLoggerHelper->log('module disabled');
                return false;
            }

            $this->shippingLoggerHelper->log('module enabled');

            $destCountry = $request->getDestCountryId();
            $destRegion = $request->getDestRegionCode();
            //TODO:fix this
            $destCity = "";

            $destRegionId = $request->getDestRegionId();

            $destPostCode = $request->getDestPostcode();

            $this->shippingLoggerHelper->log("Destination: COUNTRY: " . $destCountry . " - REGION ID: " . $destRegionId . " - REGION CODE: " . $destRegion . " - POSTCODE: " . $destPostCode);


            $shipmentConfigurationMode = $this->shippingHelper->getShipmentConfigurationMode();


            $shipments = [];

            $baseTotalInclTax = 0;
            $allItems = $request->getAllItems();
            foreach ($allItems as $item) {
                /** @var $item Item */
                // fix for wrongly calculated $item->baseTaxAmount() when the coupon is just applied
                $baseTaxAmount = $item->getTaxPercent() / 100 * ($item->getBaseRowTotal() - $item->getBaseDiscountAmount());
                $baseTotalInclTax += $baseTaxAmount + $item->getBaseRowTotal() - $item->getBaseDiscountAmount();
            }

            $this->shippingLoggerHelper->log("SHIPMENTCONFIGURATIONMODE: " . $shipmentConfigurationMode);

            $boxesDimensionWeight = [];
            $boxesSingleParcelDimensionWeight = [];

            if ($shipmentConfigurationMode == self::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_ITEM) {
                $iteration = 1;
                foreach ($request->getAllItems() as $item) {
                    if ($item->getParentItemId() == null) {
                        $itemQty =  $item->getQty();
                        $boxesDimensionWeight = [];
                        $boxesSingleParcelDimensionWeight = [];

                        // Retrieve the product info using the new box structure
                        $this->shippingHelper->getBoxesArray(
                            $boxesDimensionWeight,
                            $boxesSingleParcelDimensionWeight,
                            $item->getWeight(),
                            $this->shippingHelper->getPackageInfo($item->getSku())
                        );

                        for ($i = 1; $i <= $itemQty; $i++) {
                            $this->shippingLoggerHelper->log("Product Iteration: " . $iteration);
                            if ($this->shippingHelper->getShipmentsInsuranceMode() == \Mbe\Shipping\Helper\Data::MBE_INSURANCE_WITH_TAXES) {
                                $insuranceValue = $item->getRowTotalInclTax() / $itemQty;
                            } else {
                                $insuranceValue = $item->getRowTotal() / $itemQty;
                            }

                            // $boxesDimensionWeight is used directly, since we use 1 box for each shipment
                            $shipments = $this->getRates(
                                $destCountry, $destRegion, $destCity, $destPostCode, $baseTotalInclTax, $boxesDimensionWeight, 1, $shipments, $iteration, $insuranceValue
                            );

                            $iteration++;
                        }
                    }
                }
            } elseif ($shipmentConfigurationMode == self::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_SINGLE_PARCEL) {
//                $maxPackageWeight = $this->>shippingHelper->getMaxPackageWeight();
                $boxesDimensionWeight = [];
                $boxesSingleParcelDimensionWeight = [];
                $insuranceValue = 0;

                foreach ($request->getAllItems() as $item) {
                    $packageInfo = $this->shippingHelper->getPackageInfo($item->getSku());

                    if ($item->getParentItemId() == null) {
                        $itemQty = $item->getQty();

                        for ($i = 1; $i <= $itemQty; $i++) {
                            $boxesDimensionWeight = $this->shippingHelper->getBoxesArray(
                                $boxesDimensionWeight,
                                $boxesSingleParcelDimensionWeight,
                                $item->getWeight(),
                                $packageInfo
                            );
                        }
                        $insuranceValue += $this->shippingHelper->getSubtotalForInsurance($item);
                    }
                }

                $boxesDimensionWeight = $this->shippingHelper->mergeBoxesArray(
                    $boxesDimensionWeight,
                    $boxesSingleParcelDimensionWeight
                );

                $numBoxes = $this->shippingHelper->countBoxesArray($boxesDimensionWeight);

                $this->shippingLoggerHelper->log("Num Boxes: " . $numBoxes);

                $shipments = $this->getRates(
                    $destCountry, $destRegion, $destCity, $destPostCode, $baseTotalInclTax, $boxesDimensionWeight, $numBoxes, [], 1, $insuranceValue
                );
            } elseif ($shipmentConfigurationMode == self::SHIPMENT_CONFIGURATION_MODE_ONE_SHIPMENT_PER_SHOPPING_CART_MULTI_PARCEL) {
                $numBoxes = 0;
                $insuranceValue = 0;

                foreach ($request->getAllItems() as $item) {
                    if ($item->getParentItemId() == null) {
                        $itemQty = $item->getQty();
                        $this->shippingHelper->getSubtotalForInsurance($item);
                        $numBoxes += $itemQty;
                        for ($i = 1; $i <= $itemQty; $i++) {
                            $this->shippingHelper->getBoxesArray(
                                $boxesDimensionWeight,
                                $boxesSingleParcelDimensionWeight,
                                $item->getWeight(),
                                $this->shippingHelper->getPackageInfo($item->getSku(), true)
                            );
                        }
                    }
                }

                $this->shippingLoggerHelper->log("Num Boxes: " . $numBoxes);
                // $boxesSingleParcelDimensionWeight is used directly, since we always use 1 box for each item (we're not using packages CSV)
                $shipments = $this->getRates(
                    $destCountry, $destRegion, $destCity, $destPostCode, $baseTotalInclTax, $boxesSingleParcelDimensionWeight, $numBoxes, [], 1, $insuranceValue
                );

            }

            //TODO- remove subzone if it is the same for all shipping methods

            $subZones = [];
            foreach ($shipments as $shipment) {
                if (!in_array($shipment->subzone_id, $subZones)) {
                    array_push($subZones, $shipment->subzone_id);
                }
            }

            $useSubZone = false;
            if (count($subZones) > 1) {
                $useSubZone = true;
            }

            if (empty($shipments)) {
                $errorTitle = 'Unable to retrieve shipping methods';
                $error = $this->quoteQuoteAddressRateResultErrorFactory->create();
                $error->setCarrier($this->_code);
                $error->setCarrierTitle($this->getConfigData('title'));
                $error->setErrorMessage($errorTitle);
                $error->setErrorMessage($this->getConfigData('specificerrmsg'));
                $result->append($error);
            } else {

                foreach ($shipments as $shipment) {
                    if ($useSubZone) {
                        $currentRate = $this->_getRate($shipment->title_full, $shipment->shipment_code, $shipment->price);
                    } else {
                        $currentRate = $this->_getRate($shipment->title, $shipment->shipment_code, $shipment->price);
                    }

                    $result->append($currentRate);
                }
            }

            return $result;
        }
    }

    private function getRates(
        $destCountry, $destRegion, $city, $destPostCode, $baseTotalInclTax, $weight, $boxes, $oldResults = [], $iteration = 1, $insuranceValue
    ) {
        $this->shippingLoggerHelper->log("getRates");
        $ws = $this->shippingWsFactory->create();

        $result = [];
        $newResults = [];

        if ($this->shippingRatesHelper->useCustomRates($destCountry)) {
            $shipments = $this->shippingRatesHelper->getCustomRates($destCountry, $destRegion, $city, $destPostCode, $this->shippingHelper->getTotalWeight($weight), $insuranceValue);
        } else {
            $shipments = $ws->estimateShipping($destCountry, $destRegion, $destPostCode, $weight, $boxes, $insuranceValue);
        }

        $this->shippingLoggerHelper->logVar($shipments, 'ws estimateShipping result');

        if ($shipments) {
            $allowedShipmentServicesArray = $this->shippingHelper->getAllowedShipmentServicesArray();

            foreach ($shipments as $shipment) {
                //TODO: check if is sufficient use service
                $shipmentMethod = $shipment->Service;

                $shipmentMethodKey = $shipment->Service . "_" . $shipment->IdSubzone;

                if (in_array($shipmentMethod, $allowedShipmentServicesArray)) {
                    $shipmentTitle = __($shipment->ServiceDesc);
                    $shipmentTitle .= " - " . __($shipment->SubzoneDesc);

                    $shipmentPrice = $shipment->NetShipmentTotalPrice;

                    $shipmentPrice = $this->applyFee($shipmentPrice, $boxes);

                    $shippingThreshold = $this->shippingHelper->getThresholdByShippingServrice(
                        $shipmentMethod.($this->shippingHelper->getCountry()===$destCountry ?'_dom' :'_ww')
                    );
                    if ($shippingThreshold != null && $baseTotalInclTax >= $shippingThreshold) {
                        $shipmentPrice = 0;
                    }

                    //TODO: check if is necessary to save in some mode the courier data

                    $customLabel = $this->shippingHelper->getShippingMethodCustomLabel($shipment->Service);
                    $current = new \stdClass();
                    $current->title = $customLabel? $customLabel : (__($shipment->ServiceDesc));
                    $current->title_full = $shipmentTitle;
                    $current->method = $shipmentMethod;
                    $current->price = $shipmentPrice;
                    $current->subzone = $shipment->SubzoneDesc;
                    $current->subzone_id = $shipment->IdSubzone;
                    $current->shipment_code = $shipmentMethodKey;


                    $newResults[$shipmentMethodKey] = $current;
                }
            }

            if ($iteration == 1) {
                $result = $newResults;
            } else {
                foreach ($newResults as $newResultKey => $newResult) {
                    if (array_key_exists($newResultKey, $oldResults)) {
                        $newResult->price += $oldResults[$newResultKey]->price;
                        $result[$newResultKey] = $newResult;
                    }

                }
            }

        }

        return $result;
    }

    /**
     * Returns Allowed shipping methods
     *
     * @return array
     */
    public function getAllowedMethods()
    {
        return [
            'standard' => 'Standard delivery',
            'express' => 'Express delivery',
        ];
    }

    //
    /**
     * Get rate object based on name and price
     *
     * @param $title string Title name
     * @param $method string Method name
     * @param $price float Shipping Method price
     * @return Method
     */
    protected function _getRate($title, $method, $price)
    {
        $price = $this->shippingHelper->round($price);

        $rate = $this->quoteQuoteAddressRateResultMethodFactory->create();
        $rate->setCarrier($this->_code);
        $rate->setCarrierTitle(__($this->getConfigData('title')));
        $rate->setMethod($method);
        $rate->setMethodTitle($title);
        $rate->setPrice($price);
        $rate->setCost(0);
        return $rate;
    }


    public function applyFee($value, $packages = 1)
    {
        $handlingType = $this->getConfigData("handling_type");
        $handlingAction = $this->getConfigData("handling_action");
        $handlingFee = $this->getConfigData("handling_fee");

        if ($handlingAction == self::HANDLING_TYPE_PER_SHIPMENT) {
            $packages = 1;
        }

        if (self::HANDLING_TYPE_FIXED == $handlingType) {
            //fixed
            $result = $value + $handlingFee * $packages;
        }
        else {
            //percent
            $result = $value * (100 + $handlingFee) / 100;
        }

        return $result;
    }

    public function isTrackingAvailable()
    {
        return true;
    }

    public function getTrackingInfo($tracking)
    {
        $mbeUrl = $this->shippingTrackingHelper->getTrackingUrlBySystem($this->shippingHelper->getCountry());
        $track = $this->shippingTrackingResultStatusFactory->create();

        $track->setUrl($mbeUrl . $tracking)
            ->setTracking($tracking)
            ->setCarrierTitle($this->getConfigData('title'));
        return $track;
    }
}
