<?php
namespace Mbe\Shipping\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Mbe\Shipping\Classes\MbeWs;
use Mbe\Shipping\Helper\Data;

class Ws
{
    /** @var $ws MbeWs */
    private $ws;

    /**
     * @var \Mbe\Shipping\Helper\Data
     */
    protected $helper;

    /**
     * @var \Mbe\Shipping\Helper\Logger
     */
    protected $logger;

    /**
     * @var \Magento\Sales\Model\Order\ShipmentRepository
     */
    protected $shipment;

    /**
     *
     * @var \Magento\Framework\Filesystem\DirectoryList
     */
    protected $_dir;

    protected $scopeConfig;
    protected $writerConfig;
    protected $serializer;

    protected $wsUrl;
    protected $wsUsername;
    protected $wsPassword;
    protected $system;

    public $customer;

    public function __construct(
        \Mbe\Shipping\Helper\Data $helper,
        \Mbe\Shipping\Helper\WsLogger $logger,
        \Magento\Sales\Model\Order\ShipmentRepository $shipment,
        \Magento\Framework\Filesystem\DirectoryList $dir,
        ScopeConfigInterface $scopeConfig,
        WriterInterface $writeConfig,
        SerializerInterface $serializer,
        MbeWs $mbeWs
    ) {
        $this->shipment = $shipment;
        $this->helper = $helper;
        $this->logger = $logger;
        $this->_dir = $dir;
        $this->scopeConfig = $scopeConfig;
        $this->writerConfig = $writeConfig;
        $this->serializer = $serializer;

        $this->ws = $mbeWs;
        $this->wsUrl = $this->helper->getWsUrl();
        $this->wsUsername = $this->helper->getWsUsername();
        $this->wsPassword = $this->helper->getWsPassword();
        $this->system = $this->helper->getCountry();
        $this->customer = $this->getCustomer();
    }

    public function getCustomer()
    {
        $result = false;

        $data = $this->scopeConfig->getValue(Data::XML_PATH_CACHEDCUSTOMERDATA);
        if (!empty($data) && $data !=='false') {
            $result = $this->arrayToObject($this->serializer->unserialize($data));
        } else {
            if ($this->wsUrl && $this->wsUsername && $this->wsPassword) {
                $result = $this->ws->getCustomer($this->wsUrl, $this->wsUsername, $this->wsPassword, $this->system);
                $this->logger->logVar($result, 'WS getCustomer');
                $this->writerConfig->save(
                    Data::XML_PATH_CACHEDCUSTOMERDATA,
                    $this->serializer->serialize($result)
                );
            }
        }

        return $result;
    }

    public function arrayToObject($array)
    {
        if (is_array($array)) {
            return (object) array_map([__CLASS__,'arrayToObject'], $array);
        } else {
            return $array;
        }
    }

    public function getCustomerPermission($permissionName)
    {
        $result = null;
        $customer = $this->customer;
        if ($customer) {
            $result = $customer->Permissions->$permissionName;
        }

        return $result;
    }

    public function getAvailableOptions()
    {
        $result = false;
        if ($this->wsUrl && $this->wsUsername && $this->wsPassword) {
            $result = $this->ws->getAvailableOptions($this->wsUrl, $this->wsUsername, $this->wsPassword);
        }

        return $result;
    }

    public function getAvailableShipmentServices()
    {
        $result = array();
        $customer = $this->customer;
        if ($customer && $customer->Enabled) {
            if (isset($customer->Permissions->enabledServices)) {
                $enabledServices = $customer->Permissions->enabledServices;
                $enabledServicesDesc = $customer->Permissions->enabledServicesDesc;

                $enabledServicesArray = explode(",", $enabledServices);
                $enabledServicesDescArray = explode(",", $enabledServicesDesc);

                for ($i = 0; $i < count($enabledServicesArray); $i++) {
                    $service = $enabledServicesArray[$i];
                    $serviceDesc = $enabledServicesDescArray[$i];

                    $serviceDesc .= ' (' . $service . ')';

                    $currentShippingType = array(
                        'value' => $service,
                        'label' => $serviceDesc
                    );

                    if (!in_array($currentShippingType, $result)) {
                        array_push($result, $currentShippingType);
                    }

                    //SHIPPING WITH INSURANCE
                    if (isset($customer->Permissions->canSpecifyInsurance) && $customer->Permissions->canSpecifyInsurance) {
                        $currentShippingWithInsuranceType = array(
                            'value' => $this->helper->convertShippingCodeWithInsurance($service),
                            'label' => $this->helper->convertShippingLabelWithInsurance($serviceDesc),
                        );
                        if (!in_array($currentShippingWithInsuranceType, $result)) {
                            array_push($result, $currentShippingWithInsuranceType);
                        }
                    }
                }
            }
        }
        return $result;
    }

    public function getLabelFromShipmentType($shipmentCode)
    {
        $result = $shipmentCode;
        $allowedShipmentServices = $this->getAvailableShipmentServices();
        foreach ($allowedShipmentServices as $allowedShipmentService) {
            if ($allowedShipmentService["value"] == $shipmentCode) {
                $result = $allowedShipmentService["label"];
                break;
            }
        }
        return $result;
    }

    private function convertInsuranceShipping($shippingList)
    {
        $result = false;
        if ($shippingList) {
            $newShippingList = array();
            foreach ($shippingList as $shipping) {
                if ($shipping->InsuranceAvailable) {
                    $newShipping = $shipping;
                    $newShipping->Service = $this->helper->convertShippingCodeWithInsurance($newShipping->Service);
                    $newShipping->ServiceDesc = $this->helper->convertShippingLabelWithInsurance($newShipping->ServiceDesc);
                    array_push($newShippingList, $newShipping);
                }
            }
            if (!empty($newShippingList)) {
                $result = $newShippingList;
            }
        }
        return $result;
    }

    public function estimateShipping($country, $region, $postCode, $weight, $boxes, $insuranceValue)
    {
        $this->logger->log('ESTIMATESHIPPING');
//        $weight = $this->helper->convertWeight($weight);

        $result = false;

        if ($this->wsUrl && $this->wsUsername && $this->wsPassword) {
            $items = $this->setItems($weight);

            $shipmentType = $this->helper->getDefaultShipmentType();

            $this->logger->logVar($items, 'ESTIMATESHIPPING ITEMS');

            //Shipping without insurance
            $resultWithoutInsurance = $this->ws->estimateShipping(
                $this->wsUrl, $this->wsUsername, $this->wsPassword, $shipmentType, $this->system, $country, $region, $postCode, $items, false, $insuranceValue
            );

            //Shipping with insurance
            $resultWithInsurance = $this->ws->estimateShipping(
                $this->wsUrl, $this->wsUsername, $this->wsPassword, $shipmentType, $this->system, $country, $region, $postCode, $items, true, $insuranceValue
            );
            $resultWithInsurance = $this->convertInsuranceShipping($resultWithInsurance);

            if ($resultWithInsurance && $resultWithoutInsurance) {
                $result = array_merge($resultWithInsurance, $resultWithoutInsurance);
            }
            else {
                if ($resultWithInsurance) {
                    $result = $resultWithInsurance;
                }
                if ($resultWithoutInsurance) {
                    $result = $resultWithoutInsurance;
                }
            }
        }

        return $result;
    }


    public function createShipping(
        $country, $region, $postCode, $weight, $boxes, $products, $service, $subzone, $firstName, $lastName,$companyName, $address, $phone,
        $city, $email, $uapShipment, $goodsValue = 0.0, $reference = "", $isCod = false, $codValue = 0.0, $insurance = false, $insuranceValue = 0.0, $comment = null
    )
    {
        $this->logger->log('CREATE SHIPPING');
//        $weight = $this->helper->convertWeight($weight);

        $result = false;
        if ($this->wsUrl && $this->wsUsername && $this->wsPassword) {
            $items = $this->setItems($weight);

            $shipmentType = $this->helper->getDefaultShipmentType();


            $this->logger->logVar($items, 'CREATE SHIPPING ITEMS');

            $shipperType = $this->getShipperType();
            $result = $this->ws->createShipping(
                $this->wsUrl, $this->wsUsername, $this->wsPassword, $shipmentType, $service, $subzone, $this->system, '', $firstName, $lastName, $companyName,
                $address, $phone, $city, $region, $country, $postCode, $email, $items, $products, $uapShipment, $shipperType, $goodsValue, $reference,
                $isCod, $codValue, $insurance, $insuranceValue, $comment, $this->helper->getPluginVersion()
            );
        }
        return $result;
    }

    public function getShipperType()
    {
        //COURIERLDV or MBE
        $customer = $this->customer;
        $shipperType = "MBE";
        if ($customer->Permissions->canCreateCourierWaybill) {
            $shipperType = "COURIERLDV";
        }

        return $shipperType;
    }

    public function mustCloseShipments()
    {
        $result = true;
        $customer = $this->customer;
        if (!$customer || $customer->Permissions->canCreateCourierWaybill) {
            $result = false;
        }
        return $result;
    }

    public function getCustomerMaxParcelWeight()
    {
        $customer = $this->customer;
        return $customer->Permissions->maxParcelWeight;
    }

    public function getCustomerMaxShipmentWeight()
    {
        $customer = $this->customer;
        return $customer->Permissions->maxShipmentWeight;
    }

    public function isCustomerActive()
    {
        $result = false;
        $customer = $this->customer;
        if ($customer) {
            $result = $customer->Enabled;
        }
        return $result;
    }

    public function closeShipping(array $shipmentIds)
    {
        $trackingNumbers = array();
        foreach ($shipmentIds as $shipmentId) {
            $shipment = $this->shipment->get($shipmentId);
            $tracks = $shipment->getTracksCollection()->getColumnValues('track_number');
            foreach ($tracks as $track) {
                $trackingNumber = $track->getTrackNumber();
                array_push($trackingNumbers, $trackingNumber);
            }
        }

        $this->closeTrackingNumbers($trackingNumbers);
    }

    public function closeTrackingNumbers(array $trackingNumbers)
    {
        $this->logger->log('CLOSE SHIPPING');

        $result = false;

        if ($this->wsUrl && $this->wsUsername && $this->wsPassword) {
            $result = $this->ws->closeShipping($this->wsUrl, $this->wsUsername, $this->wsPassword, $this->system, $trackingNumbers);
            if($result) {
                foreach ($trackingNumbers as $trackingNumber) {
                    $filePath = $this->helper->getTrackingFilePath($trackingNumber);
                    file_put_contents($filePath, $result->Pdf);
                }
            }
        }

        return $result;
    }

    public function setItems($boxesWeight)
    {
        $items = [];
        foreach ($boxesWeight as $box) {
            foreach ($box['weight'] as $weight) {
                $item = new \stdClass;
                $item->Weight = $this->helper->convertWeight($weight);
                $item->Dimensions = new \stdClass;
                $item->Dimensions->Lenght = $box['dimensions']['length'];
                $item->Dimensions->Height = $box['dimensions']['height'];
                $item->Dimensions->Width = $box['dimensions']['width'];
                $items[] = $item;
            }
        }
        return $items;
    }
    public function returnShipping($toReturnIds)
    {
        $result = [];
        foreach ($toReturnIds as $toReturnId) {
            $shipment = $this->shipment->get($toReturnId);
            $tracks = $shipment->getTracksCollection()->getColumnValues('track_number');
            if (!empty($tracks)) {
//  			foreach ($tracks as $track) {
//	    			array_push($trackingNumbers, $track);
//		    	}
//      		return $this->returnTrackingNumbers($trackingNumbers);
                $result[$toReturnId] = $this->returnTrackingNumbers([$tracks[0]]); // TODO : only manage the 1st order's shipment for now
            }
        }
        return $result;
    }

    public function returnTrackingNumbers(array $trackingNumbers)
    {
        $this->logger->logVar( $trackingNumbers, 'RETURN SHIPPING ');

        $returnTracking = [];

        if ($this->wsUrl && $this->wsUsername && $this->wsPassword) {
            foreach ( $trackingNumbers as $trackingNumber ) {
                $result = $this->ws->returnShipping($this->wsUrl, $this->wsUsername, $this->wsPassword, $this->system, $trackingNumber, $this->helper->getPluginVersion());
                if (!empty($result)) {
                    $returnTracking[$trackingNumber] = $result->MasterTrackingMBE;
                } else {
                    // error on creation of the return shipment
                    $returnTracking[$trackingNumber] = false;
                }
            }
        }
        return $returnTracking;
    }
}