<?php

namespace Sartoric\ModularCms\Traits;

use Illuminate\Support\Facades\Validator;
use Sartoric\ModularCms\Helpers\SlugHelper;
use Sartoric\ModularCms\Models\Slug;
use Illuminate\Validation\ValidationException;

/* Class parameter $slugField can be defined in Model class to set the field to be used as the slug value */
/* Class parameter $slugShowController can be defined in Model class to set the function (usually a controller)
   to run to show the entity */
// Route::get('/{locale}/{slug}', 'SlugController@show')->name('page');
// parameter slug_entity_id will be added to the request forwarded to $slugShowController

trait Slugable
{

    public $isSlugable = true;

    /**
     * Define the entity's field to be used as the slug value (can be null)
     * @return string|null
     */
    abstract protected function getSlugFieldAttribute();

    /**
     * Define the action (usually a controller one) to run in order to show the entity
     * @return string
     */
    abstract protected function getSlugShowControllerAttribute();

    /**
     * Delete slug when entity gets deleted
     */
    protected static function bootSlugable()
    {
        // Delete slug row when the entity has been deleted
        self::deleted(function ($entity) {
            $entity->slugRel()->delete();
        });
    }

    /**
     * Set the relation to the slug table
     * @return mixed
     */
    public function slugRel()
    {
        return $this->morphOne(Slug::class, 'slugable');
    }

    public function gethasSlugAttribute()
    {
        return isset($this->slugRel->slug);
    }

    /**
     * Set the slug field in the Slug model using the polymorphic relation.
     * If no value is passed, the field set as parameter $slugField will be used to retrieve the value from the model
     * If checkOk is set to false the slugOk check is discarded and duplicates can be inserted (check to be done in
     * model's controller)
     * *
     * @param null $value
     * @param bool $checkDuplicate
     * @return bool
     */
    public function setSlug($value = null, $checkDuplicate = true)
    {
        try {
            Validator::make(['slug' => $value], [
                'slug' => 'alpha_dash|max:255|nullable',
            ])->validate();

            // get slug value from model if $value is empty
            if ((!isset($value) || $value==='') && (isset($this->slugField) && $this->slugField !== '')) {
                $value = $this->{$this->slugField};
            }

            // set the slug value if it's valid
            if ((isset($value) && $value !=='') && SlugHelper::slugOk($value, $this, $checkDuplicate)) {
                $this->slugRel()->updateOrCreate(
                    ['slugable_id' => $this->getKey()],
                    ['slug' => $value]
                );
                return true;
            }
            return false;
        } catch (ValidationException $err) {
            return false;
        }
    }
}
