<?php

namespace Sartoric\ModularCms\Traits;

use Sartoric\ModularCms\Helpers\TranslationHelper;
use Sartoric\ModularCms\Models\Setting;
use Sartoric\ModularCms\Models\Translation;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

/**
 * Trait Translatable
 * Enable the translations for the model
 * ** Frontend parameters **
 * (required) public parameter $translationShowController is used to forward the show request to a specific controller after retrieving the translation info
 * (required) public parameter $translationRedirectController is used to forward the request and run a custom redirect
 * ** Backend parameters **
 * (required) public parameter $translationEditRoute is used to redirect to a specific route after the creation of a new entity's translation (eg. 'page.edit')
 * (required) public parameter $translationDelete is used to call the entity deletion
 * public parameter $translationCloner is used to run a custom record duplication when creating a new entity's translation
 * @package App\Traits
 */
trait Translatable
{
    public $hasTranslation = true;
    abstract protected function setTranslationShowControllerAttribute();
    abstract protected function settranslationRedirectControllerAttribute();
    abstract protected function setTranslationEditRouteAttribute();
    abstract protected function setTranslationDeleteAttribute();

    /**
     * Delete translation when model gets deleted
     */
    protected static function bootTranslatable()
    {
        // Delete translation row when the model has been deleted
        self::deleted(function ($entity) {
            $entity->translationRel()->delete();
        });
    }

    public function getIsTranslationAttribute()
    {
        return !empty($this->translationRel->id_main);
    }

    public function getMainTranslations()
    {
        return self::whereHas('translationRel', function (Builder $query) {
            $query->where('id_main', '=', null);
        });
    }

    public function getParentTranslation()
    {
        $idMain = $this->translationRel->id_main;
        if (!empty($idMain)) {
            $mainTranslation = clone $this; // create an object of the same type
            return $mainTranslation::find($idMain)->translationRel->id;
        } else {
            return $this->translationRel->id;
        }
    }

    public function getTranslatedItem($locale = null)
    {
        $cookieLocale = Str::camel(Setting::appSetting('appname')) . '.' . TranslationHelper::FRONTEND_LANGUAGE_COOKIE;
        $locale = $locale ?: session($cookieLocale); // TODO Valutare se usare app.locale
        $translation = $this->translationList();
        return $this::find($translation[TranslationHelper::languageIso2Id($locale)]);
    }

    /**
     * @return array, Return all the translations associated to the item [Language id=>Item id]
     */
    public function translationList()
    {
        if ($this->translationRel) {
            if (empty($this->translationRel->id_main)) {
                $list = Translation::where('id_main', $this->getKey())
                    ->where('translatable_type', get_class($this))
                    ->pluck('translatable_id', 'id_language')->all(); // retrieve all the childrens
                $list = [$this->translationRel->id_language => $this->getKey()] + $list; // add itself
                // TODO : verificare languages()
                $languages = array_fill_keys(TranslationHelper::languages()->pluck('id')->all(), null); //retrieve all the languages
                $list = $list + $languages; // merge the arrays keeping the keys
            } else {
                // TODO: Controllare se serve anda e rianda
                // $list = Translation::find($this->translationRel()->id_main)->translatable->translationList(); // retrieve all the brothers (and itself) page and language
                $list = $this::find($this->translationRel->id_main)->translationList();
            }
            return $list;
        }
        return array();
    }

    /**
     * @param $language - Language iso code
     * @param $mainId - null when setting the main
     * @return bool
     */
    public function setAsTranslation($language, $mainId = null)
    {
        $langId = TranslationHelper::languageIso2Id($language);
        try {
            // set polymorphic
            DB::transaction(function () use (&$translation, $mainId, $langId) {
                $translation = $this->translationRel()->updateOrCreate(
                    ['id_main' => $mainId],
                    ['id_language' => $langId],
                    ['translatable_id' => $this->getKey()]
                );
            });
            return $translation->id;
        } catch (Exception $err) {
            return false;
        }
    }

    public function languageName() {
        if($this->translationRel) {
            return TranslationHelper::languageName($this->translationRel->id_language);
        }
        return null;
    }

    /**
     * Set the relation to the slug table
     * @return mixed
     */
    public function translationRel()
    {
        return $this->morphOne(Translation::class, 'translatable');
    }
}
